# starsim_control.py
# David Handy  19 October 2004
"""
StarSim start/stop control program
"""

import ConfigParser
import os
import re
import socket
import StringIO
import subprocess
import sys
import threading
import time
import Tkinter as Tk
import tkMessageBox
import traceback

import cpif
from cpif.graphics import centerWindow
from cpif.rpc import connect, Disconnected, TimeoutException

import starsim
import starsim.const
import starsim.scenarios
from starsim.player.playermod import Player
from starsim.server.simulator import Simulator

CPIF_MIN_VERSION = (1, 0, 2)


def makeVersionTuple(s, num_parts=None):
    m = re.match(r"(\d+(\.\d+)*)(.*)", s)
    result = []
    if m:
        for part in m.group(1).split('.'):
            try:
                result.append(int(part))
            except ValueError:
                break
        if m.group(3):
            result.append(m.group(3))
    if num_parts is None:
        num_parts = len(result)
    return tuple(result[:num_parts] + ([0]*max(0, num_parts - len(result))))

module = sys.modules[__name__]
script_file = getattr(module, '__file__', sys.argv[0])
script_dir = os.path.dirname(script_file)

help_url = os.path.join(script_dir, 'starsim.html')
help_url = os.path.normpath(help_url)
help_url = os.path.normcase(help_url)
help_url = os.path.abspath(help_url)
help_url= 'file://' + help_url


class ErrorMessage(Tk.Toplevel):
    """
    An error box that displays a failure message with details available
    by pressing a button.
    """

    def __init__(self, root, operation, cause, details):
        """
        root: the parent window
        operation: the name of the attempted operation that failed
        cause: short string giving the cause of the failure
        details: longer message, possibly a traceback, related to the failure
        """
        Tk.Toplevel.__init__(self, root)
        self.__details = details
        self.__details_message = None # the Message widget showing details
        #self.transient(root)
        self.title("Error")
        Tk.Label(self,
                 text="Operation failed: " + operation).pack(
                    side=Tk.TOP, expand=Tk.YES, fill=Tk.BOTH)
        Tk.Label(self,
                 text="Cause of failure: " + cause).pack(
                    side=Tk.TOP, expand=Tk.YES, fill=Tk.BOTH)
        button_frame = Tk.Frame(self)
        self.__details_button = Tk.Button(button_frame, text="Show Details",
                  command=self.__showDetails)
        self.__details_button.pack(
                      side=Tk.LEFT, expand=Tk.YES, fill=Tk.BOTH)
        Tk.Button(button_frame, text="Quit",
                  command=self.__quit).pack(
                      side=Tk.LEFT, expand=Tk.YES, fill=Tk.BOTH)
        button_frame.pack(
            side=Tk.TOP, expand=Tk.YES, fill=Tk.BOTH)

    def __showDetails(self):
        if self.__details_message:
            self.__details_message.destroy()
            self.__details_message = None
            self.__details_button.configure(text="Show Details")
        else:
            #self.__details_message = Tk.Message(self, text=self.__details)
            self.__details_message = Tk.Label(self, text=self.__details)
            self.__details_message.pack(
                side=Tk.TOP, expand=Tk.YES, fill=Tk.BOTH)
            self.__details_button.configure(text="Hide Details")
        
    def __quit(self):
        self.destroy()


def trycallTk(root, operation_name, func, args=(), kwargs={}):
    """
    Call a function or bound method and display an error message window
    with details if it raises an exception.

    root: the root Tk window to use for the message window, or None.
    operation_name: the display name of the attempted operation
    func: the function or bound method to call
    args: the parameters to pass to the function or bound method
    kwargs: the keyword parameters to pass to the function or bound method

    Return the return value of the function or bound method if there was
    no exception raised; otherwise return None.
    """
    try:
        return func(*args, **kwargs)
    except:
        traceback.print_exc()
        # Get the exception info
        e = sys.exc_info()[1] # the exception value
        tbtext = StringIO.StringIO()
        traceback.print_exc(file=tbtext)
        sys.exc_clear()
        # Create the message window
        msg = ErrorMessage(root, operation_name, str(e), tbtext.getvalue())
        root.wait_window(msg)
        return None


def getStarSimDataDir():
    # If the script directory looks like it has the shared files
    # then use that directory, otherwise use the fixed directory.
    if os.path.exists(os.path.join(script_dir, 'starsim_server.ini')):
        return script_dir
    else:
        return starsim.getDataDir()


def testTCPConnect(addr):
    s = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
    s.settimeout(10.0) # seconds
    try:
        s.connect(addr)
        s.close() # connection suceeded
        return True, "Ok"
    except socket.error, e:
        msg = (str(e) + "\n" +
               "The server may not be running,\n" +
               "or the host name or port number may be incorrect.")
        return False, msg


def getPortFromINI(config, section, option, default=None, verbose=False):
    """
    Read a positive integer from a ConfigParser instance.
    Return the default value if the section or option is missing or invalid.
    If verbose is true, then also print a message on warnings.
    """
    try:
        port_str = config.get(section, option)
    except ConfigParser.Error:
        if verbose:
            print >> sys.stderr, "Warning: [%s] %s: not found in .ini file" % (
                section, option),
            print >> sys.stderr, "(expected a port number)"
        return default
    try:
        port_num = int(port_str)
        if port_num <= 0 or port_num >= 65535:
            raise ValueError()
        return port_num
    except ValueError:
        print >> sys.stderr, "Error: [%s] %s = %s:" % (section, option,
                                                       port_str),
        print >> sys.stderr, "(not a valid port number)"
        return default


class ChoiceButtonGroup:
    """
    A group of "radio" (choice) buttons

    Call addButton() once for each button, in display order.
    Call create() once to display the buttons.
    Call ok() sometime later to do the button action.
    """
    
    def __init__(self, parent_widget):
        """
        parent_widget:
            The parent widget that will own this button group.  If
            parent_widget is not a "real" widget (i.e, it's a
            DrawingWindow), then its window or frame attribute is used for
            the parent_widget instead.
        """
        if isinstance(parent_widget, (Tk.Widget, Tk.Toplevel, Tk.Tk)):
            self.parent_widget = parent_widget
        else:
            self.parent_widget = parent_widget.window or parent_widget.frame
        self.frame = None
        self.choice_var = None
        self.choices = {} # { id : (callback, text) }
        self.cur_id = 0

    def addButton(self, callback, text):
        """
        Add another button to the choices.
        Return the ID of the button added.

        callback:
            Function or bound method to be called when this choice is
            selected and Ok is called.
        text:
            Text that serves as the button label.
        """
        self.cur_id += 1
        self.choices[self.cur_id] = (callback, text)
        return self.cur_id

    def create(self):
        """
        Create the buttons that were defined by calling addButton().
        Only call this method once per button group.
        """
        self.frame = Tk.Frame(self.parent_widget)
        self.choice_var = Tk.IntVar(self.frame)
        keys = self.choices.keys()
        keys.sort()
        for key in keys:
            text = self.choices[key][1]
            b = Tk.Radiobutton(self.frame, text=text, value=key,
                               variable=self.choice_var)
            b.pack(side=Tk.TOP, anchor=Tk.W)
        self.frame.pack()

    def destroy(self):
        if self.frame:
            self.frame.destroy()

    def ok(self):
        """
        Do the action associated with the selected button.
        Return the ID of the selected button, or None if no button was
        selected.
        """
        choice = self.choice_var.get()
        if not choice:
            callback = None
        else:
            callback = self.choices.get(choice, (None, None))[0]
        if callback:
            callback()
        return choice

    def select(self, button_id):
        """
        Select one of the buttons. Call this after calling create().

        button_id: The ID of the button, returned by addButton()
        """
        if self.choice_var:
            self.choice_var.set(button_id)


class MessageHandler:

    def __init__(self, connection):
        self._connection = connection

def _nullErrorHandler(connection, error):
    pass


class StarSimController:

    server_section = 'StarSimServer'
    player_section = 'StarSimPlayer'

    DIALOG_DONE_TAG = '<<SubDialog.done>>'
    SERVER_ERROR_TAG = '<<ServerError>>'
    SERVER_EXIT_TAG = '<<ServerExit>>'

    def __init__(self):
        self.cur_subdialog_instance = None
        self.next_subdialog = None
        self.simulator = None
        self.server_port = 9000
        self.server_addr = None # (host, port) or rpc Node instance
        self.server_thread = None
        self.inproc = False
        self.fallback_port_range_start = 9001
        self.fallback_port_range_end = 9099
        self.module_name = starsim.const.DEFAULT_SCENARIO
        self.file_name = 'game1.dat'
        self.player_host = None
        self.player_port = 9000
        self.server_command = None
        self.user_dir = None
        #####
        self.root = Tk.Tk()
        self.root.withdraw() # keep root hidden
        self.root.bind(self.DIALOG_DONE_TAG, self.subDialogDone)
        self.root.bind(self.SERVER_ERROR_TAG, self.serverErrorCallback)
        self.root.bind(self.SERVER_EXIT_TAG, self.serverExitCallback)
        self.startSubDialog(MainMenu)

    def createServerSocket(self, verbose=True):
        """
        Create a server socket for listening for incoming connections.
        Return the socket object, ready for calling accept(), or the
        error instance if the socket could not be created and bound.
        """
        try:
            s = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        except socket.error, e:
            if verbose:
                print >> sys.stderr, e
            return e
        def bind(s, addr):
            if verbose:
                print "binding to", addr; sys.stdout.flush()
            try:
                s.bind(addr)
                s.listen(5)
                return None
            except socket.error, e:
                if verbose:
                    print >> sys.stderr, e
                return e
        self.server_addr = ('', self.server_port)
        error_inst = bind(s, self.server_addr)
        if not error_inst:
            return s
        for port in xrange(self.fallback_port_range_start,
                           self.fallback_port_range_end+1):
            addr = ('', port)
            if not bind(s, addr):
                self.server_port = port
                self.server_addr = addr
                return s
        return error_inst

    def sendCommand(self, method, params, timeout=5.0):
        """
        Connect to the RPC server, send one admin command, then disconnect.

        method: Name of method to call on server, i.e. 'ping' or 'stop'
        params: Tuple of parameters for method, () if no parameters.
        timeout: Seconds to wait for response before raising TimeoutException

        Typically if there is a communication failure, this function raises a
        Disconnected exception.
        """
        if self.inproc:
            simulator = self.simulator
            if not simulator:
                raise Disconnected("Simulator not created yet")
            node = simulator.node
            if not node:
                raise Disconnected("Simulator node not created yet")
            addr = node
        else:
            addr = self.server_addr
            host, port = addr
            if not host:
                host = 'localhost'
                addr = (host, port)
        connection = connect(addr, MessageHandler, timeout=timeout,
                             errorHandler=_nullErrorHandler)
        proxy = connection.getBlockingProxy(timeout)
        try:
            status, msg = proxy.hello(starsim.const.ADMIN_USER)[:2]
            if not status:
                print "Admin login failed:", msg
                return None
            return connection.callMethodAndBlock(method, params, timeout)
        finally:
            connection.close()

    def destroy(self):
        root = self.root
        self.root = None
        if root:
            root.destroy()

    def getUserDir(self):
        if not self.user_dir:
            self.user_dir = trycallTk(self.root, 
                                 "Getting the user's StarSim data directory", 
                                 starsim.getUserDir)
            if self.user_dir is None:
                self.user_dir = ''
        return self.user_dir

    def loadServerDefaults(self):
        global_ini = self.getGlobalServerINIFile()
        user_ini = self.getUserServerINIFile()
        ini_files = [global_ini, user_ini]
        print "reading", ini_files; sys.stdout.flush()
        cp = ConfigParser.SafeConfigParser()
        cp.read(ini_files)
        # Port
        self.server_port = getPortFromINI(cp, self.server_section,
                                          'server_port',
                                          default=self.server_port)
        self.fallback_port_range_start = \
            getPortFromINI(cp, self.server_section, 'fallback_port_range_start',
                           default=self.fallback_port_range_start)
        self.fallback_port_range_end = \
            getPortFromINI(cp, self.server_section, 'fallback_port_range_end',
                           default=self.fallback_port_range_end)
        # Module
        try:
            module_name = cp.get(self.server_section, 'module_name')
        except ConfigParser.Error:
            module_name = None
        if module_name:
            self.module_name = module_name
        # File
        try:
            file_name = cp.get(self.server_section, 'file_name')
        except ConfigParser.Error:
            file_name = None
        if file_name:
            self.file_name = file_name

    def loadPlayerDefaults(self):
        global_ini = self.getGlobalPlayerINIFile()
        user_ini = self.getUserPlayerINIFile()
        ini_files = [global_ini, user_ini]
        print "reading", ini_files; sys.stdout.flush()
        cp = ConfigParser.SafeConfigParser()
        cp.read(ini_files)
        # Host
        try:
            player_host = cp.get(self.player_section, 'player_host')
        except ConfigParser.Error:
            player_host = None
        if player_host:
            self.player_host = player_host
        # Port
        self.player_port = getPortFromINI(cp, self.player_section,
                                          'player_port',
                                          default=self.player_port)

    def getGlobalServerINIFile(self):
        return os.path.join(script_dir, 'starsim_server.ini')

    def getUserServerINIFile(self):
        return os.path.join(self.getUserDir(), 'starsim_server.ini')

    def getGlobalPlayerINIFile(self):
        return os.path.join(script_dir, 'starsim_player.ini')

    def getUserPlayerINIFile(self):
        return os.path.join(self.getUserDir(), 'starsim_player.ini')

    def runServer(self, server_socket):
        """
        Run the StarSim server.

        server_socket -- A listening TCP socket object, or None for an
                         in-process single-player game
        """
        try:
            self.simulator = Simulator()
            if self.server_command == 'new':
                print "Creating new StarSim game"
                self.simulator.create(self.module_name, self.file_name)
            elif self.server_command == 'load':
                print "Loading StarSim game"
                self.simulator.load(self.file_name,)
            self.simulator.run(server_socket)
        except:
            traceback.print_exc()
            self.root.after(0, self.serverErrorCallback)
            return
        self.simulator.printStats()
        if self.root:
            self.root.event_generate(self.SERVER_EXIT_TAG)

    def startPlayer(self):
        if self.inproc:
            self.startInProcPlayer()
            return
        if self.player_host <> 'localhost':
            # only save remote settings
            self.savePlayerSettings()
        print "Starting player and connecting to", self.player_host,
        print "on port", self.player_port
        starsim_player = os.path.join(script_dir, 'starsim_player.py')
        if not os.path.exists(starsim_player):
            tkMessageBox.showerror("Error",
                                   "Could not find file '" +
                                   starsim_player + "'")
            self.destroy()
            return
        exe_dir, exe_name = os.path.split(sys.executable)
        if not exe_name.lower().startswith('python'):
            # If this script was started by a py2exe executable, use a
            # regular python interpreter in the same directory to launch
            # the player script.
            exe_name = 'python'
        exe_path = os.path.join(exe_dir, exe_name)
        def escape(param):
            if ' ' in param:
                return '"%s"' % param
            else:
                return param
        cmd = [exe_path, starsim_player, self.player_host, 
                str(self.player_port)]
        print ' '.join([escape(part) for part in cmd])
        subprocess.call(cmd)

    def startInProcPlayer(self, window=None):
        if not self.simulator or not self.simulator.is_running:
            raise Exception("Can't start Player until Simulator is running")
        if not window:
            window = Tk.Toplevel(self.root)
        player = Player(self.simulator.node, help_url=help_url, window=window)
        player.connect()
        window.update()
    
    def startServer(self):
        if self.inproc:
            server_socket = None
        else:
            server_socket = self.createServerSocket()
            if isinstance(server_socket, socket.error):
                tkMessageBox.showerror(title="Error",
                   message="""\
Couldn't start the server on port %s or on ports %s-%s due to an error:

%s

(The server may already be running, or another program may already be using
those ports, or networking may not be set up properly on this machine.)
""" % (self.server_port, self.fallback_port_range_start, 
       self.fallback_port_range_end, server_socket))
                return False
        self.saveServerSettings()
        if self.player_host == 'localhost':
            print "startServer: setting player_port to", self.server_port
            self.player_port = self.server_port
        self.server_thread = threading.Thread(target=self.runServer,
                                              args=(server_socket,))
        self.server_thread.setDaemon(True) # I didn't pick the name...
        self.server_thread.start()
        return True

    def isServerRunning(self):
        if self.simulator:
            return self.simulator.is_running
        return False

    def startSubDialog(self, next_subdialog):
        name = getattr(next_subdialog, '__name__', '(SubDialog)')
        instance = trycallTk(self.root, name, next_subdialog, (self,))
        if not instance:
            # Dialog constructor failed - exit or we'll be stuck
            self.destroy()
        else:
            self.cur_subdialog_instance = instance

    def subDialogDone(self, event):
        print 'subDialogDone',
        if self.cur_subdialog_instance:
            print self.cur_subdialog_instance.__class__.__name__,
            self.cur_subdialog_instance.window.destroy()
            self.cur_subdialog_instance = None
        next_subdialog = self.next_subdialog
        if not next_subdialog:
            # Our job is done
            self.destroy()
            return
        print '->', next_subdialog.__name__
        self.startSubDialog(next_subdialog)

    def saveServerSettings(self):
        # Assuming these settings have already been validated
        cp = ConfigParser.SafeConfigParser()
        cp.add_section(self.server_section)
        cp.set(self.server_section, 'server_port', str(self.server_port))
        cp.set(self.server_section, 'fallback_port_range_start',
               str(self.fallback_port_range_start))
        cp.set(self.server_section, 'fallback_port_range_end',
               str(self.fallback_port_range_end))
        cp.set(self.server_section, 'module_name', self.module_name)
        cp.set(self.server_section, 'file_name', self.file_name)
        starsim_server_ini = self.getUserServerINIFile()
        print "writing", starsim_server_ini
        f = file(starsim_server_ini, 'w')
        print >> f, "; StarSim server initialization file"
        print >> f, "; Keeps track of the last settings that you used"
        print >> f
        cp.write(f)
        print >> f
        print >> f, "; end-of-file"
        f.close()

    def savePlayerSettings(self):
        # Assuming these settings have already been validated
        cp = ConfigParser.SafeConfigParser()
        cp.add_section(self.player_section)
        cp.set(self.player_section, 'player_host', self.player_host)
        cp.set(self.player_section, 'player_port', str(self.player_port))
        starsim_player_ini = self.getUserPlayerINIFile()
        print "writing", starsim_player_ini
        f = file(starsim_player_ini, 'w')
        print >> f, "; StarSim player initialization file"
        print >> f, "; Keeps track of the last settings that you used when",
        print >> f, "connecting to a remote computer."
        print >> f
        cp.write(f)
        print >> f
        print >> f, "; end-of-file"
        f.close()

    def serverErrorCallback(self, event=None):
        tkMessageBox.showerror(title="Error",
            message="Help! The server quit unexpectedly!\n"
                    "(It's probably a bug.)\n"
                    "Check the console window for error messages.")
        self.destroy()

    def serverExitCallback(self, event=None):
        print "Normal server exit"
        self.destroy()

    def cleanup(self):
        """
        Called when self.root.mainloop() exits. Cleans up anything necessary
        before the program exits. (i.e. stops the simulator if it is
        running).
        """
        if self.simulator and self.simulator.is_running:
            self.simulator.stop()
            while self.simulator.is_running:
                print "Waiting for simulator to stop..."
                time.sleep(1.0)


class SubDialog:

    def __init__(self, controller, text=''):
        self.controller = controller
        self._done_called = False
        self.window = Tk.Toplevel(controller.root)
        self.window.pack_propagate(True) # book says this is default
        self.window.title("StarSim control")
        self.frame = Tk.Frame(self.window)
        label = Tk.Label(
            self.frame, 
            text="StarSim game control")
        label.pack(side=Tk.TOP)
        self.label = Tk.Label(self.frame, text=text)
        self.label.pack(side=Tk.TOP, anchor=Tk.W)
        self.window.bind('<Destroy>', self.onDestroy)

    def createOkCancel(self):
        """
        Create the standard Ok and Cancel buttons
        """
        frame = Tk.Frame(self.frame)
        b = Tk.Button(frame, text="Ok", command=self.ok)
        b.pack(side=Tk.LEFT)
        b = Tk.Button(frame, text="Cancel", command=self.cancel)
        b.pack(side=Tk.LEFT)
        frame.pack(side=Tk.TOP)

    def cancel(self):
        self.controller.destroy()

    def done(self):
        self._done_called = True
        self.controller.root.event_generate(self.controller.DIALOG_DONE_TAG)

    def onDestroy(self, event):
        if not self._done_called and str(event.widget) == str(self.window):
            self.controller.destroy()

    def ok(self):
        raise NotImplementedError() # "Abstract" method

    def setMessage(self, message):
        self.label.configure(text=message)


class MainMenu(SubDialog):

    def __init__(self, controller):
        SubDialog.__init__(self, controller, text="Choose an action below:")
        self.choice_group = ChoiceButtonGroup(self.frame)
        choices = [
            (self.startStandAloneGame, 'Start a single-player game'),
            (self.startServerAndPlayer, 'Start a multi-player game'),
            (self.startPlayer, 'Connect to a multi-player game'),
            #(self.startServer, 'Run the game server only, no player yet'),
            ]
        for callback, text in choices:
            id = self.choice_group.addButton(callback, text)
        self.choice_group.create()
        self.createOkCancel()
        self.frame.pack()
        centerWindow(self.window)
        # Check cpif version (Anyone know a better place to do this?)
        cpif_version = makeVersionTuple(cpif.__version__)
        print "cpif version:", cpif_version
        print "cpif minimum version:", CPIF_MIN_VERSION
        if cpif_version < CPIF_MIN_VERSION:
            tkMessageBox.showerror(title="Error",
                message="The cpif package version is %s.\n"
                        "The minimum required version is %s." % 
                            (cpif_version, CPIF_MIN_VERSION))
            self.controller.next_subdialog = None
            self.done()

    def startStandAloneGame(self):
        self.controller.loadServerDefaults()
        self.controller.loadPlayerDefaults()
        self.controller.inproc = True
        if not self.controller.player_host:
            self.controller.player_host = 'localhost'
        self.controller.next_subdialog = ChooseNewOrOldGame
        self.done()

    def startServerAndPlayer(self):
        self.controller.loadServerDefaults()
        self.controller.loadPlayerDefaults()
        self.controller.player_host = 'localhost'
        #self.controller.next_subdialog = GetServerPort
        self.controller.next_subdialog = ChooseNewOrOldGame
        self.done()

    def startPlayer(self):
        self.controller.loadServerDefaults()
        self.controller.loadPlayerDefaults()
        self.controller.next_subdialog = ChooseLocalOrRemoteConnect
        self.done()

    def startServer(self):
        self.controller.loadServerDefaults()
        #self.controller.next_subdialog = GetServerPort
        self.controller.next_subdialog = ChooseNewOrOldGame
        self.done()

    def ok(self):
        choice_key = self.choice_group.ok()
        print "MainMenu.ok", choice_key
        if not choice_key:
            tkMessageBox.showwarning(title="No way, try again",
                                     message="Please make a choice first")
            return


class GetServerPort(SubDialog):

    def __init__(self, controller):
        SubDialog.__init__(self, controller, 
           text="Enter a port number for the StarSim server to use")
        self.createBody()
        self.createOkCancel()
        self.frame.pack()
        centerWindow(self.window)

    def createBody(self):
        frame = Tk.Frame(self.frame)
        Tk.Label(frame, text='Port number:').grid(row=0, column=0,
                                                   sticky=Tk.W)
        self.server_port_var = Tk.StringVar(frame)
        self.server_port_var.set(str(self.controller.server_port))
        entry = Tk.Entry(frame, width=16, textvariable=self.server_port_var)
        entry.grid(row=0, column=1, sticky=Tk.W)
        frame.pack()

    def ok(self):
        server_port_str = self.server_port_var.get()
        print "GetServerPort.ok", server_port_str
        try:
            server_port = int(server_port_str)
        except ValueError:
            server_port = None
        if server_port is None or server_port <= 0:
            msg = '"%s" is not a valid port number.' % server_port_str
            tkMessageBox.showwarning(title="No way, try again", message=msg)
            return
        self.controller.server_port = server_port
        if self.controller.player_host:
            self.controller.player_port = server_port
        self.controller.next_subdialog = ChooseNewOrOldGame
        self.done()


class ChooseLocalOrRemoteConnect(SubDialog):

    def __init__(self, controller):
        msg = "Where is the StarSim game server that you wish to use?"
        SubDialog.__init__(self, controller, text=msg)
        self.choice_group = ChoiceButtonGroup(self.frame)
        choices = [
            (self.connectRemote, 'On another computer', True),
            (self.connectLocal, 'On this computer', False),
            ]
        default_choice = None
        for callback, text, default in choices:
            id = self.choice_group.addButton(callback, text)
            if default:
                default_choice = id
        self.choice_group.create()
        if default_choice:
            self.choice_group.select(default_choice)
        self.createOkCancel()
        self.frame.pack()
        centerWindow(self.window)

    def connectRemote(self):
        print "ChooseLocalOrRemoteConnect.connectRemote"
        self.controller.next_subdialog = GetPlayerHost

    def connectLocal(self):
        print "ChooseLocalOrRemoteConnect.connectLocal"
        self.controller.player_host = 'localhost'
        self.controller.player_port = self.controller.server_port
        self.controller.next_subdialog = GetPlayerPort

    def ok(self):
        chosen = self.choice_group.ok()
        print "ChooseLocalOrRemoteConnect.ok", chosen
        if not chosen:
            tkMessageBox.showwarning(title="No way, try again",
                                     message="Please make a choice first")
            return
        self.done()


class GetPlayerHost(SubDialog):

    def __init__(self, controller):
        msg = ("Which computer is running the StarSim server?\n" +
               "Enter the computer name or IP address.")
        SubDialog.__init__(self, controller, text=msg)
        self.createBody()
        self.createOkCancel()
        self.frame.pack()
        centerWindow(self.window)

    def createBody(self):
        frame = Tk.Frame(self.frame)
        Tk.Label(frame, text='Computer name or IP address:').grid(
            row=0, column=0, sticky=Tk.W)
        self.host_name_var = Tk.StringVar(frame)
        player_host = self.controller.player_host
        if player_host is None:
            player_host = ''
        self.host_name_var.set(player_host)
        entry = Tk.Entry(frame, width=16, textvariable=self.host_name_var)
        entry.grid(row=0, column=1, sticky=Tk.W)
        frame.pack()

    def ok(self):
        host_name = self.host_name_var.get().strip()
        print "GetPlayerHost.ok", host_name
        if not host_name:
            msg = "I can't connect to a server with no name."
            tkMessageBox.showwarning(title="No way, try again", message=msg)
            return
        self.controller.player_host = host_name
        self.controller.next_subdialog = GetPlayerPort
        self.done()


class GetPlayerPort(SubDialog):

    def __init__(self, controller):
        SubDialog.__init__(self, controller, 
           text="Enter a port number of the StarSim server")
        self.createBody()
        self.createOkCancel()
        self.frame.pack()
        centerWindow(self.window)

    def createBody(self):
        frame = Tk.Frame(self.frame)
        Tk.Label(frame, text='Port number:').grid(row=0, column=0,
                                                   sticky=Tk.W)
        self.player_port_var = Tk.StringVar(frame)
        self.player_port_var.set(str(self.controller.player_port))
        entry = Tk.Entry(frame, width=16, textvariable=self.player_port_var)
        entry.grid(row=0, column=1, sticky=Tk.W)
        frame.pack()

    def ok(self):
        player_port_str = self.player_port_var.get().strip()
        print "GetPlayerPort.ok", player_port_str
        try:
            player_port = int(player_port_str)
        except ValueError:
            player_port = None
        if player_port is None or player_port <= 0:
            msg = '"%s" is not a valid port number.' % player_port_str
            tkMessageBox.showwarning(title="No way, try again", message=msg)
            return
        player_host = self.controller.player_host
        status, error = testTCPConnect((player_host, player_port))
        if not status:
            msg = ('Cannot connect to port %d on "%s":\n'
                    '%s' % (player_port, player_host, error))
            tkMessageBox.showerror(title="Can't contact server", message=msg)
            return
        self.controller.player_port = player_port
        self.controller.next_subdialog = None
        self.controller.startPlayer()
        self.done()


class ChooseNewOrOldGame(SubDialog):

    def __init__(self, controller):
        msg = "Start a new game or load an old one?"
        SubDialog.__init__(self, controller, text=msg)
        self.createChoiceButtons()
        self.createOkCancel()
        self.frame.pack()
        centerWindow(self.window)

    def createChoiceButtons(self):
        choices = [
            (self.newGame, "Start a new game"),
            (self.loadGame, "Load an old game"),
            ]
        self.choice_group = ChoiceButtonGroup(self.frame)
        for callback, text in choices:
            self.choice_group.addButton(callback, text)
        self.choice_group.create()

    def newGame(self):
        self.controller.server_command = 'new'

    def loadGame(self):
        self.controller.server_command = 'load'

    def ok(self):
        self.choice_group.ok()
        print "ChooseNewOrOldGame.ok", self.controller.server_command
        if self.controller.server_command == 'new':
            self.controller.next_subdialog = GetGameFile # GetGameModule
        elif self.controller.server_command == 'load':
            self.controller.next_subdialog = GetGameFile
        else:
            tkMessageBox.showwarning(title="No way, try again",
                                     message="Please make a choice first")
            return
        self.done()


class GetGameModule(SubDialog):

    def __init__(self, controller):
        msg = "Which universe module would you like to use?"
        SubDialog.__init__(self, controller, text=msg)
        self.createBody()
        self.createOkCancel()
        self.frame.pack()
        centerWindow(self.window)

    def createBody(self):
        frame = Tk.Frame(self.frame)
        Tk.Label(frame, text='Module name:').grid(row=0, column=0,
                                                   sticky=Tk.W)
        self.module_name_var = Tk.StringVar(frame)
        self.module_name_var.set(self.controller.module_name)
        entry = Tk.Entry(frame, width=16, textvariable=self.module_name_var)
        entry.grid(row=0, column=1, sticky=Tk.W)
        frame.pack()
        frame = Tk.Frame(self.frame)
        self.module_list = starsim.scenarios.listmodules()
        self.module_index_var = Tk.IntVar(frame)
        self.module_listbox = Tk.Listbox(frame,
                                         listvariable=self.module_index_var,
                                         height=10, width=16)
        scrollbar = Tk.Scrollbar(frame, command=self.module_listbox.yview)
        self.module_listbox.configure(yscrollcommand=scrollbar.set)
        self.module_listbox.pack(side=Tk.LEFT)
        scrollbar.pack(side=Tk.RIGHT, fill=Tk.Y)
        for item in self.module_list:
            self.module_listbox.insert(Tk.END, item)

    def ok(self):
        module_name = self.module_name_var.get()
        if not module_name:
            msg = "I can't start the server without a module name."
            tkMessageBox.showwarning(title="No way, try again", message=msg)
            return
        print "GetGameModule.ok", module_name
        self.controller.module_name = module_name
        self.controller.next_subdialog = GetGameFile
        self.done()


NEW_GAME_MSG = "Choose the name of the file in which the game will be saved"
LOAD_GAME_MSG = "Choose the file from which the game data will be loaded"

class GetGameFile(SubDialog):

    def __init__(self, controller):
        msg = "<Oops, no message>"
        if controller.server_command == 'new':
            msg = NEW_GAME_MSG
        elif controller.server_command == 'load':
            msg = LOAD_GAME_MSG
        SubDialog.__init__(self, controller, text=msg)
        self.createBody()
        self.createOkCancel()
        self.frame.pack()
        centerWindow(self.window)

    def createBody(self):
        frame = Tk.Frame(self.frame)
        Tk.Label(frame, text='File name:').grid(row=0, column=0, stick=Tk.W)
        self.file_name_var = Tk.StringVar(frame)
        self.file_name_var.set(self.controller.file_name)
        entry = Tk.Entry(frame, width=16, textvariable=self.file_name_var)
        entry.grid(row=0, column=1, sticky=Tk.W)
        frame.pack()

    def ok(self):
        file_name = self.file_name_var.get()
        print "GetGameFile.ok", file_name
        file_name = self.checkFileName(file_name)
        if file_name:
            self.controller.file_name = file_name
            self.controller.next_subdialog = StartServer
            self.done()

    def checkFileName(self, file_name):
        """Check and transform a file name.
        Return the file name to be used if Ok, None otherwise."""
        if not file_name:
            tkMessageBox.showwarning(title="No way, try again",
                message="You have to pick a file name first.")
            return None
        file_name = os.path.join(self.controller.getUserDir(), file_name)
        file_name = os.path.normpath(file_name)
        file_name = os.path.normcase(file_name)
        if not file_name.endswith('.dat'):
            file_name = file_name + '.dat'
        if os.path.isdir(file_name):
            tkMessageBox.showerror(title="Error",
                message='"%s" is the name of a directory.\n'
                        'Select a file name instead.' % file_name)
            return None
        if self.controller.server_command == 'load':
            if not os.path.isfile(file_name):
                tkMessageBox.showerror(title="Error",
                    message='"%s" is not a valid file name.' % file_name)
                return None
            size = os.path.getsize(file_name)
            if not size:
                tkMessageBox.showerror(title="Error",
                    message='"%s" is a zero-length file.' % file_name)
                return None
        else:
            # command is assumed to be 'new'
            if os.path.exists(file_name):
                result = tkMessageBox.askyesno(title="Wait a second...",
                    message='"%s" is the name of a file that exists.\n'
                            'Do you want to erase this file?' % file_name)
                if not result:
                    return None
            try:
                f = file(file_name, 'wb')
                f.close() # create zero-length file
            except Exception, e:
                tkMessageBox.showerror(title="Error",
                    message='Could not create file "%s": %s' %
                        (file_name, str(e)))
                return None
        return file_name


class StartServer(SubDialog):

    TIMEOUT_SEC = 30

    def __init__(self, controller):
        msg = "Waiting for StarSim server to finish starting ..."
        SubDialog.__init__(self, controller, text=msg)
        self.frame.pack()
        centerWindow(self.window)
        self.start_time = time.time()
        if not self.controller.startServer():
            # startServer() should have shown an error message
            self.cancel()
            return
        self.window.after(1000, self.checkServerStart)

    def checkServerStart(self):
        if self.controller.isServerRunning():
            self.serverIsRunning()
            return
        if time.time() - self.start_time > self.TIMEOUT_SEC:
            msg = "Server failed to start after %s seconds" % self.TIMEOUT_SEC
            tkMessageBox.showerror(title="Problem", message=msg)
            self.cancel()
            return
        self.window.after(1000, self.checkServerStart)

    def serverIsRunning(self):
        if self.controller.inproc:
            self.controller.next_subdialog = StartInProcPlayer
        else:
            self.controller.next_subdialog = RunServer
        self.done()


class RunServer(SubDialog):

    def __init__(self, controller):
        server_host = socket.gethostname()
        server_msg = """\
Computer name: %(server_host)s
Port number: %(server_port)d""" % {
        'server_host': server_host, 
        'server_port': controller.server_port,
        }
        if controller.player_host:
            player_msg = "The Player window should start soon.\n"
        else:
            player_msg = ""
        msg = """\
The StarSim server is running

%(server_msg)s

%(player_msg)s
Press Stop to quit the server and save the game.
Game file: %(file_name)s""" % {
            'server_msg': server_msg,
            'player_msg': player_msg,
            'file_name': controller.file_name}
        SubDialog.__init__(self, controller, text=msg)
        self.stop_b = Tk.Button(self.frame, text="Stop", command=self.stop)
        self.stop_b.pack(side=Tk.TOP)
        self.frame.pack()
        centerWindow(self.window)
        if controller.player_host:
            controller.startPlayer()

    def stop(self):
        self.controller.next_subdialog = StopServer
        self.done()


class StopServer(SubDialog):

    def __init__(self, controller):
        msg = "Waiting for server to shut down..."
        SubDialog.__init__(self, controller, text=msg)
        self.frame.pack()
        centerWindow(self.window)
        self.window.bind('<<ServerStopped>>', self.serverIsStopped)
        t = threading.Thread(target=self.stopServer)
        t.setDaemon(True) # Really, I don't like that name...
        t.start()

    def serverIsStopped(self, event=None):
        self.controller.next_subdialog = None
        self.done()

    def stopServer(self):
        try:
            try:
                self.controller.sendCommand('stop', ())
            except (Disconnected, TimeoutException):
                # Server is already down, or at least not communicating
                pass
        finally:
            self.window.event_generate('<<ServerStopped>>')


class StartInProcPlayer(SubDialog):
    """
    Start the Player window in single-player mode
    (this dialog window becomes the player window)
    """

    def __init__(self, controller):
        SubDialog.__init__(self, controller, text='')
        controller.startInProcPlayer(self.window)


def main():
    ssc = StarSimController()
    if ssc.root:
        ssc.root.mainloop()
    ssc.cleanup()
    print "StarSim controller done."


if __name__ == '__main__':
    main()

# end-of-file
