# simulator.py
# David Handy  16 Oct 2004
"""
starsim game simulator
"""

import random
import sys
import time

from cpif.rpc import Listener, Disconnected, InProcNode
from starsim.server import model, universedb
from starsim.const import SCAN_LIMIT, TIME_PERIOD, ADMIN_USER
from starsim import util

ALLOWED_BACKLOG = 5


class BaseMessageHandler(object):

    def __init__(self, connection, simulator):
        self._connection = connection
        self._simulator = simulator

    def exit(self):
        user = getattr(self, '_user', None)
        if user:
            print "Received exit message from", user
        else:
            print "Received exit message"
        self._connection.close()
        self._simulator.handleExit(self)

    def ping(self, data):
        """
        Return data back to the player (client) to let it know we are alive.
        """
        return data


class StartingMessageHandler(BaseMessageHandler):

    def hello(self, user):
        """
        Declare your user ID to the simulator. One ID is special, and allows
        one to send admin commands.

        Return (status, msg), where status is true if the user ID was
        accepted, and msg is a displayable string.
        """
        print "Received hello from", repr(user)
        connection = self._connection
        simulator = self._simulator
        if user == ADMIN_USER:
            connection.setHandler(AdminMessageHandler(connection, simulator))
            return (True, "You may now send admin commands.")
        status, msg = simulator.handleHello(connection, user)
        if status:
            connection.setHandler(PlayerMessageHandler(connection, simulator, 
                                                       user))
        return (status, msg)


class AdminMessageHandler(BaseMessageHandler):

    def save(self, file_name=None):
        print "Received save command"
        self._simulator.save(file_name)

    def stats(self):
        self._simulator.printStats()

    def stop(self):
        print "Received stop command"
        self._simulator.stop()


class CommandMethod:

    def __init__(self, target_obj, command_name):
        self.target_obj = target_obj
        self.command_name = command_name

    def __call__(self, *params):
        self.target_obj.postCommand(self.command_name, params)


class PlayerMessageHandler(BaseMessageHandler):

    def __init__(self, connection, simulator, user):
        super(PlayerMessageHandler, self).__init__(connection, simulator)
        self._user = user
        self._proxy = connection.getProxy(None) # "no-reply" proxy
        self._ready = False
        self._commands = 0
        self._updates = 0
        self._skips = 0
        self._backlog = 0
        self.ship = None

    def __getattr__(self, name):
        # Any method not defined here is delegated to the user's ship
        if name.startswith('_'):
            raise AttributeError("%s: insecure attribute access" % name)
        if not self.ship:
            raise AttributeError("%s: no ship to command" % name)
        self._commands += 1
        return CommandMethod(self.ship, name)

    def queryInitInfo(self, obj_ids):
        """
        Return the list of object initializers for obj_ids.
        """
        return self._simulator.universe.queryInitInfo(obj_ids)

    def requestShip(self, ship_id=None):
        """
        Request permission to control a ship.

        ship_id: The ID of a particular ship the user might want.

        Return (ship_id, msg):
            ship_id: ship ID if sucessful, None otherwise
            msg: Message to show the user
        """
        self.ship, msg = self._simulator.requestShip(self._user, 
                                                     ship_id=ship_id)
        print self._user, "assigned to ship #", self.ship.id
        return (self.ship.id, msg)

    def ready(self):
        self._ready = True

    def _update(self, data):
        """Send update back to the player."""
        if not self._ready:
            self._backlog += 1
            if self._backlog > ALLOWED_BACKLOG:
                self._skips += 1
                return
        else:
            self._backlog = 0
            self._ready = False
        try:
            self._proxy.update(data)
            self._updates += 1
        except Disconnected:
            self._simulator.handleExit(self)

    def _sendUserMessage(self, msgs):
        """Send messages back to the player."""
        try:
            self._proxy.showMessages(msgs)
        except Disconnected:
            self._simulator.handleExit(self)

    def _printStats(self):
        print "Statistics for", self, ":"
        print "Controlling user           :", self._user
        print "Ship ID                    :", self.ship and self.ship.id
        print "Number of commands received:", self._commands
        print "Number of updates processed:", self._updates
        print "Number of updates skipped  :", self._skips


class HandlerFactory:

    def __init__(self, simulator):
        self._simulator = simulator

    def __call__(self, connection):
        return StartingMessageHandler(connection, self._simulator)


class Simulator:

    def __init__(self):
        self.node = None
        self.is_running = False
        self.max_clients = 0
        self.universe = None
        self.user_connection_map = {} # { user: connection }
        self.single_step = False
        self.file_name = None

    def create(self, module_name, file_name):
        self.file_name = file_name
        self.universe = universedb.create(module_name)
        assert self.universe
        return True

    def handleDisconnect(self, connection):
        self.handleExit(connection.getHandler())

    def handleExit(self, client):
        user = getattr(client, '_user', None)
        if not self.user_connection_map.pop(user, None):
            # user was already disconnected
            return
        if user:
            print "User", user, "disconnected."
        if client and hasattr(client, '_printStats'):
            client._printStats()

    def handleHello(self, connection, user):
        """
        Handle a new connection.

        Return (status, msg):
            status: true iff the user is accepted
            msg: Message to show the user
        """
        if user in self.user_connection_map:
            return (False, "Sorry, %s is already logged in." % (user,))
        self.user_connection_map[user] = connection
        if len(self.user_connection_map) > self.max_clients:
            self.max_clients = len(self.user_connection_map)
        return (True, "Welcome, " + user)

    def load(self, file_name):
        self.file_name = file_name
        self.universe = universedb.load(file_name)
        return True

    def printStats(self):
        print "Server statistics:"
        print "Max. number of clients:", self.max_clients
        for connection in self.user_connection_map.values():
            client = connection.getHandler()
            if client and isinstance(client, PlayerMessageHandler):
                client._printStats()

    def requestShip(self, user, ship_id=None):
        """
        Process a request from a user to take control of a ship.

        user:
            The ID of the requesting user
        ship_id:
            The ID of the particular ship desired, or None to request the
            simulator to pick a ship.

        Return (sim_obj, msg):
            sim_obj: The ship object controlled by the user
            msg: Message to show the user
        """
        if ship_id:
            sim_obj = self.universe.getObject(ship_id)
            if not sim_obj:
                return (None, "Sorry, that ship is gone. Try another.")
        else:
            # pick one of user's starships
            sim_obj = None
            for obj in self.universe.getObjsForUser(user):
                if isinstance(obj, model.StarShip):
                    sim_obj = obj
                    break
        if not sim_obj:
            # Create a new ship for a new user
            sim_obj = self._newStarShip(user)
        else:
            prev_user = sim_obj.user
            if prev_user and prev_user <> user:
                return (None, "Sorry, that ship was taken by %s. Try another." %
                        prev_user)
            self.universe.setObjectUser(sim_obj, user)
        return (sim_obj, 
            "Welcome aboard, captain %s. Press F1 for help." % user)

    def _newStarShip(self, user):
        x = (random.random() * SCAN_LIMIT * 2) - SCAN_LIMIT
        y = (random.random() * SCAN_LIMIT * 2) - SCAN_LIMIT
        return model.StarShip(self.universe, (x, y), user=user)

    def createNode(self, port_or_socket):
        """
        Create a listening RPC node that will accept client connections to
        this simulator.

        port_or_socket --
            a TCP port number to listen on, or
            a TCP listening socket object, or
            None to create an in-process RPC node
        """
        handlerFactory = HandlerFactory(self)
        if port_or_socket is None:
            node = InProcNode(handlerFactory)
        else:
            if type(port_or_socket) == int:
                addr = ('', port_or_socket)
            else:
                addr = port_or_socket
            node = Listener(addr, handlerFactory)
        return node

    def _isNodeLike(self, obj):
        expected_methods = ('run', 'stop', 'onDisconnectCall',
                'onErrorCall', 'dump', 'load')
        for name in expected_methods:
            if not hasattr(obj, name):
                return False
        return True

    def run(self, port_or_socket):
        """
        Run the simulator.

        port_or_socket --
            a TCP port number to listen on, or
            a TCP listening socket object, or
            None to create an in-process single-player game, or
            The return value of createNode()
        """
        if self._isNodeLike(port_or_socket):
            self.node = port_or_socket
        else:
            self.node = self.createNode(port_or_socket)
        self.node.onDisconnectCall(self.handleDisconnect)
        self.is_running = True
        t0 = util.gettime()
        loop_count = 0
        while self.is_running:
            if self.single_step:
                raw_input("Press Enter to do next update... ")
            ticks = self.universe.update()
            for connection in self.user_connection_map.values():
                client = connection.getHandler()
                sim_obj = getattr(client, 'ship', None)
                if not sim_obj:
                    continue
                display_list = sim_obj.getDisplayList()
                display_list.extend(sim_obj.getPrivateDisplayList())
                client._update((ticks, display_list))
                msgs = sim_obj.getAndClearMessages()
                if msgs:
                    client._sendUserMessage(msgs)
            loop_count += 1
            tt = t0 + (loop_count * TIME_PERIOD)
            while True:
                timeout = tt - util.gettime()
                if timeout < 0:
                    timeout = 0
                if not self.node.poll(timeout):
                    self.is_running = False
                    break
                if not timeout:
                    break

    def save(self, file_name=None):
        if file_name:
            self.file_name = file_name
        if not self.file_name:
            raise ValueError("Can't save game, no file name.")
        universedb.save(self.file_name, self.universe)

    def stop(self):
        # called by a client
        for connection in self.user_connection_map.values():
            client = connection.getHandler()
            if client and isinstance(client, PlayerMessageHandler):
                try:
                    connection.callMethod('stop', (), None)
                except Disconnected:
                    pass
            connection.close()
        self.node.stop()
        if self.file_name:
            self.save()
        self.is_running = False


if __name__ == '__main__':
    sys.exit(main())

# end-of-file
