# starsim/model.py
"""
StarSim data model

Attribute naming convention:

    Methods that implement user commands have names beginning with cmd_.
    (See the postCommand method for more details.)

"""

import math
import sys

from starsim.server import auto
from starsim.const import SCAN_LIMIT, TICKS_PER_SECOND, DOCK_DISTANCE
from starsim.util import parseNumberPair


def sin(a):
    return math.sin(a * (math.pi / 180.0))
def cos(a):
    return math.cos(a * (math.pi / 180.0))
def movepolar(x, y, a, r):
    return (x + (r * sin(a)), y + (r * cos(a)))


class SimObj(object):
    """
    Base class for all simulation objects.
    """

    # If the 'private' attribute is true, then this object will not show up in
    # ordinary sector scans.
    private = False

    # If the 'active' attribute is true, then update() will be called on
    # this object every time tick.
    active = False

    # By default, objects don't move.
    vel = (0.0, 0.0)

    def __init__(self, universe, loc):
        """
        universe: The Universe in which this object will exist
        loc: (x, y) the starting location of the object

        All derived classes that add additional parameters to __init__ must
        make them keyword parameters.
        """
        self._universe = universe
        self.loc = loc
        self.deleted = False
        self.id = universe.registerObject(self)

    def delete(self):
        """
        Delete self and all component objects. Removes them from the
        universe as well.
        """
        self._universe.deleteObject(self)
        self.deleted = True

    def update(self):
        """
        Update the state of the simulation object. This defines the object's
        behavior.
        """
        pass

    def getInitInfo(self):
        """
        Return a tuple containing the class name, the ID, and the invariant
        view information. This tuple contains the view object constructor
        parameters.
        """
        return (self.__class__.__name__, self.id) # derived class adds more

    def getPrivateDisplayList(self):
        """
        Return a sequence of tuples containing current view information for
        objects only visible by this object, i.e. destination markers.
        
        These tuples are generated by calling the getView() methods of those
        private objects.  (Private objects exist in the universe database
        but have a 'private' attribute value of True, and therefore do not
        show up in ordinary searches.)
        """
        return []

    def getView(self):
        """
        Return a tuple containing the ID, the location, and the current
        visible state of the starship. This tuple contains the parameters to
        the update() method on the view object.
        """
        return (self.id, self.loc) # derived class adds more

    # Utility methods

    def getDistanceTo(self, sim_obj):
        loc = self.loc
        dest_loc = sim_obj.loc
        deltaX = dest_loc[0] - loc[0]
        deltaY = dest_loc[1] - loc[1]
        return math.sqrt(deltaX*deltaX + deltaY*deltaY)

    def getObject(self, id):
        return self._universe.getObject(id)

    def getHomeBase(self):
        return self._universe.home_base

    def getDisplayList(self):
        return self._universe.getDisplayList(self.loc)


class Star(SimObj):
    """
    A star. Doesn't do much, mostly background ornamentation.
    """


class MovingMarker(SimObj):
    """
    Base class for special objects that mark a location or object.
    
    A marker location can't be updated for objects that are out of scan
    range, so it uses the "last known location" when something goes out of
    sight.
    """

    # This object should not show up in ordinary sector scans.
    private = True

    # This object should be updated every tick
    active = True

    def __init__(self, universe, loc, track_id=None, **init_params):
        """
        track_id:
            ID of the object this marker is attached to, or None if the
            marker is attached to a fixed location.
        """
        super(MovingMarker, self).__init__(universe, loc, **init_params)
        self.track_id = track_id

    def update(self):
        """
        Update the marker. If it has a tracking object, move its location to
        the current tracking object's location.
        """
        if self.track_id:
            track_obj = self.getObject(self.track_id)
            if not track_obj or track_obj.deleted:
                self.track_id = None
                self.delete()
            else:
                if self.getDistanceTo(track_obj) <= SCAN_LIMIT:
                    self.loc = track_obj.loc


class DestinationMarker(MovingMarker):
    """
    A destination object or location.
    """


class SelectionMarker(MovingMarker):
    """
    A destination object or location.
    """


class SmartObj(SimObj):

    """
    A SmartObj is an active simulation object that can receive commands
    and messages.
    """

    # This object should be updated every tick
    active = True

    def __init__(self, universe, loc, **init_params):
        super(SmartObj, self).__init__(universe, loc, **init_params)
        self.selection = None # command target
        self.__command_q = []
        self.__message_q = []

    def __getstate__(self):
        state = self.__dict__.copy()
        # delete unpicklable objects (if any)
        return state

    def __setstate__(self, state):
        self.__dict__.update(state)
        # restore unpicklable objects (if any)

    def delete(self):
        if self.selection:
            self.selection.delete()
            self.selection = None
        super(SmartObj, self).delete()

    def update(self):
        for method, params in self.getAndClearCommands():
            method(*params)

    def getPrivateDisplayList(self):
        private_list = super(SmartObj, self).getPrivateDisplayList()
        if self.selection:
            private_list.append(self.selection.getView())
        return private_list

    # Command posting/retreiving

    def postCommand(self, command_name, params):
        """
        Schedule a command to be executed on the next update.
        command_name -- the name of the command method minus the cmd_ prefix
        params -- the command parameters
        """
        method = getattr(self, 'cmd_' + command_name)
        self.__command_q.append((method, params))

    def getAndClearCommands(self):
        """
        Return the contents of the command queue, also clears the queue.
        Returns [(method, params)]
        """
        commands = self.__command_q
        self.__command_q = []
        return commands

    # Commands common to all SmartObj-derived objects

    def cmd_setSelection(self, id_or_loc_or_None):
        """
        Set the target of the next command.

        id_or_loc_or_None:
            If a non-zero integer ID, the ID of the object that is the
            selected object.
            If a coordinate tuple, select a location.
            If None (or zero), then clear any previous selection and don't
            set a new one.
        """
        old_sel = self.selection
        self.selection = None
        if type(id_or_loc_or_None) is tuple:
            # Parameter is a location
            loc = None
            track_id = None
            if not old_sel:
                self.postMessage("Can't select empty space")
        elif isinstance(id_or_loc_or_None, SimObj):
            # Parameter is an object
            loc = id_or_loc_or_None.loc
            track_id = id_or_loc_or_None.id
        else:
            # Parameter is an object ID
            sim_obj = self.getObject(id_or_loc_or_None)
            if sim_obj:
                loc = sim_obj.loc
                track_id = sim_obj.id
            else:
                loc = None
                track_id = None
        if loc:
            self.selection = SelectionMarker(self._universe, loc, 
                                             track_id=track_id)
        if old_sel:
            old_sel.delete()

    def cmd_sendStarGram(self, text):
        if not self.selection or isinstance(self.selection, tuple):
            self.postMessage("Can't send StarGram: select an object first")
            return
        dest_obj = self.getObject(self.selection.track_id)
        if not dest_obj:
            self.postMessage("Can't send StarGram: no destination")
            return
        self.sendStarGramTo(dest_obj, text)

    # Message posting/retreiving

    def postMessage(self, msg):
        self.__message_q.append(msg)

    def getAndClearMessages(self):
        msgs = self.__message_q
        self.__message_q = []
        return msgs

    def sendStarGramTo(self, dest_obj, text):
        sender = getattr(self, 'user', None)
        if not sender:
            sender = getattr(self, 'name', None)
        stargram = StarGram(self._universe, self.loc, angle=self.angle, 
                            vel=self.vel, va=0.0, sender=sender, content=text)
        stargram.cmd_setDestination(dest_obj)


class ContainerObj(SmartObj):

    """
    A ContainerObj is a SmartObj that can contain cargo.
    """

    DELTA = 0.000001

    def __init__(self, universe, loc, **init_params):
        super(ContainerObj, self).__init__(universe, loc, **init_params)
        self._cargo = {} # {'item-name': amount}

    def addCargo(self, item_name, amount):
        assert amount >= 0
        self._cargo[item_name] = self._cargo.get(item_name, 0) + amount

    def takeCargo(self, item_name, amount):
        assert amount >= 0
        cur_amount = self._cargo.get(item_name, 0)
        if cur_amount < amount:
            amount = cur_amount
        new_amount = cur_amount - amount
        if new_amount < self.DELTA:
            # Used up the last of the cargo
            self._cargo.pop(item_name, None)
        return amount


class NeedyObj(ContainerObj):

    """
    A NeedyObj is a ContainerObj that can have needs (desire for certain
    amounts of certain types of cargo, for stated reasons.)
    """

    NEEDS_REVIEW_INTERVAL_TICKS = TICKS_PER_SECOND * 30

    def __init__(self, universe, loc, **init_params):
        super(NeedyObj, self).__init__(universe, loc, **init_params)
        self._needs = [] # [('item-name', amount, 'reason')]
        self._needs_review_timer = self.NEEDS_REVIEW_INTERVAL_TICKS

    def update(self):
        super(NeedyObj, self).update()
        self._needs_review_timer -= 1
        if self._needs_review_timer <= 0:
            self._needs_review_timer = self.NEEDS_REVIEW_INTERVAL_TICKS
            self._broadcastNeeds()

    # Specific to a NeedyObj:

    def _broadcastNeeds(self):
        if not self._needs:
            return
        dests = [self.getHomeBase()]
        dests.extend(self.getLocalFriendlyShips())
        for item_name, amount, reason in self._needs:
            msg = ("Need %d units of %s on %s (%+5.2f, %+5.2f): %s" %
                (amount, item_name, self.name, self.loc[0], self.loc[1],
                 reason))
            for dest in dests:
                self.sendStarGramTo(dest, msg)

    def getLocalFriendlyShips(self):
        # Get list of local ships that may be friendly
        ships = []
        display_list = self.getDisplayList()
        for items in display_list:
            id = items[0]
            dest_obj = self.getObject(id)
            if not dest_obj:
                continue
            if not isinstance(dest_obj, StarShip):
                continue
            if isinstance(dest_obj, FuonInvader):
                continue
            ships.append(dest_obj)
        return ships

    def addNeed(self, item_name, amount, reason):
        assert amount >= 0
        for i in xrange(len(self._needs)):
            cur_item, cur_amount, cur_reason = self._needs[i]
            if (cur_item, cur_reason) == (item_name, reason):
                # Existing need became greater
                new_amount = cur_amount + amount
                self._needs[i] = (item_name, new_amount, reason)
                return
        # New need
        self._needs.append((item_name, amount, reason))

    def addCargo(self, item_name, amount):
        # When cargo is given to a needy object, it first meets its existing
        # needs before adding any surplus to its cargo store.
        # Delete needs that have been adequately met.
        assert amount >= 0
        new_needs = []
        for cur_item, cur_amount, cur_reason in self._needs:
            if cur_item == item_name:
                if cur_amount <= amount:
                    new_amount = 0
                    amount = amount - cur_amount
                else:
                    new_amount = cur_amount - amount
                    amount = 0
            else:
                new_amount = cur_amount
            if new_amount < self.DELTA:
                continue
            new_needs.append((cur_item, new_amount, cur_reason))
        self._needs = new_needs
        if amount > 0:
            super(NeedyObj, self).addCargo(item_name, amount)


class StarBase(NeedyObj):
    """
    A StarBase is a stationary active object that lets starships dock with
    it. It can have cargo and sometimes needs cargo.
    """

    NUM_DOCK_BAYS = 8

    def __init__(self, universe, loc, 
                 color='gray', 
                 name='',
                 **init_params):
        super(StarBase, self).__init__(universe, loc, **init_params)
        self.angle = 0.0 # angle of rotating scanner
        self.color = color
        self.name = name
        self.dock_bays = [None] * self.NUM_DOCK_BAYS

    def update(self):
        super(StarBase, self).update()
        self.angle += 360.0 / (TICKS_PER_SECOND * 8.0)

    def getInitInfo(self):
        """
        Return a tuple containing the invariant drawing info.
        """
        return super(StarBase, self).getInitInfo() + (self.color,)

    def getPrivateDisplayList(self):
        return []

    def getView(self):
        """
        Return a tuple containing the latest drawing info.
        """
        return super(StarBase, self).getView() + (self.angle, self.name)

    # StarBase-specific methods

    # Requests

    def requestDock(self, sim_obj):
        """
        Return (dock_angle, message)
        dock_angle is None if docking is refused.
        """
        if self.getDistanceTo(sim_obj) > DOCK_DISTANCE:
            return (None, "Out of docking range from " + self.name)
        for i in xrange(len(self.dock_bays)):
            if self.dock_bays[i] is None:
                self.dock_bays[i] = sim_obj
                dock_angle = (float(i) / len(self.dock_bays)) * 360.0
                return (dock_angle, "Approved to dock at " + self.name)
        return (None, "No free docking bay at " + self.name)

    def requestUnDock(self, sim_obj):
        """
        Un-dock an object. No return value (always succeeds).
        """
        try:
            i = self.dock_bays.index(sim_obj)
        except ValueError:
            return
        self.dock_bays[i] = None


class FuonBase(StarBase):

    # Override
    def requestDock(self, sim_obj):
        """
        Return (dock_angle, message)
        dock_angle is None if docking is refused.
        """
        if not issubclass(sim_obj, FuonInvader):
            return (None, "Request to dock was rudely denied")
        return super(FuonBase, self).requestDock(sim_obj)


class StarVessel(SmartObj):
    """
    A StarVessel is an active, moving object that can turn, accelerate, and
    navigate itself to a destination via its autopilot.

    A StarVessel also has a configurable main color, to better visually
    distinguish vessels of the same type.
    """

    TURN_INC = 2.0
    MAX_TURN_RATE = 90.0
    SPEED_INC = 0.0005
    MAX_SPEED = 0.005
    FRICTION = 0.05
    TURN_BRAKE_THRESH = 45.0 # degrees
    BOOST_BRAKE_THRESH = MAX_SPEED * 5.

    MISSILE_LAUNCH_SPEED = MAX_SPEED * 0.9

    def __init__(self, universe, loc,
                color='gray',    # main fill color
                angle=0.0,       # heading angle in clockwise degrees
                vel=(0.0, 0.0),  # (vx, vy) velocity, sectors/tick
                va=0.0,          # clockwise angular velocity, degrees/tick
                **init_params):
        super(StarVessel, self).__init__(universe, loc, **init_params)
        self.color = color
        self.angle = angle
        self.vel = vel # velocity: (vx, vy)
        self.va = va   # angular velocity
        self.dest = None # destination marker
        self.turn = 0.0
        self.boost = 0.0
        self.dock_station = None
        self.__auto = auto.AutoPilot(self, 
                                     self.TURN_BRAKE_THRESH,
                                     self.BOOST_BRAKE_THRESH)

    def delete(self):
        if self.dest:
            self.dest.delete()
            self.dest = None
        super(StarVessel, self).delete()

    def update(self):
        super(StarVessel, self).update()
        if self.dock_station:
            self._dockUpdate()
            return
        turn = self.turn
        self.turn = 0.0
        boost = self.boost
        self.boost = 0.0
        # get autopilot controls
        dest = self.dest
        if dest and not dest.deleted:
            auto_turn, auto_boost = self.__auto.advise(dest.loc)
            if (auto_turn, auto_boost) <> (0.0, 0.0):
                turn = (auto_turn + turn) / 2.0
                boost = (auto_boost + boost) / 2.0
            else:
                self.arrivalTrigger()
        self._turn(turn * self.TURN_INC)
        self._accel(boost * self.SPEED_INC) # sets self.vel
        vx, vy = self.vel
        loc = self.loc
        self.loc = (loc[0] + vx, loc[1] + vy)
        self.angle += self.va
        slide = (1.0 - self.FRICTION)
        self.vel = (slide * vx, slide * vy)
        self.va *= slide

    def getInitInfo(self):
        """
        Return a tuple containing the class name, the ID, and the invariant
        view information. This tuple contains the view object constructor
        parameters.
        """
        return super(StarVessel, self).getInitInfo() + (self.color,)

    def getPrivateDisplayList(self):
        private_list = super(StarVessel, self).getPrivateDisplayList()
        if self.dest:
            private_list.append(self.dest.getView())
        return private_list

    def getView(self):
        """
        Return a tuple containing the ID, the location, and the current
        visible state of the starship. This tuple contains the parameters to
        the update() method on the view object.
        """
        return super(StarVessel, self).getView() + (self.angle,)

    # Special StarVessel stuff

    def arrivalTrigger(self):
        # Called when this object arrives at its destination
        # Most subclasses will override this
        pass

    # Commands

    def cmd_home(self):
        self.cmd_setDestination(self.getHomeBase())

    def cmd_left(self):
        self.turn = -1.0

    def cmd_right(self):
        self.turn = 1.0

    def cmd_up(self):
        self.boost = 1.0

    def cmd_down(self):
        self.boost = -1.0

    def cmd_dock(self):
        """
        Dock with destination object.
        Post a display message indicating the status of the command.
        """
        if self.dock_station:
            self._unDock()
            return
        marker = self.dest
        if not marker:
            self.postMessage("Cannot dock: no destination selected")
            return
        target_id = marker.track_id
        if not target_id:
            self.postMessage("Cannot dock with empty space")
            return
        target_obj = self.getObject(target_id)
        if not target_obj:
            self.postMessage("Cannot dock: no destination object")
            return
        if not hasattr(target_obj, 'requestDock'):
            self.postMessage("Selected object is not dockable")
            return
        dock_angle, msg = target_obj.requestDock(self)
        self.postMessage(msg)
        if dock_angle is None:
            return
        self.dock_station = (target_obj, dock_angle)
        self.cmd_setDestination(None)

    def cmd_setDestination(self, id_or_loc_or_None):
        """
        Set the StarVessel's current destination.

        id_or_loc_or_None:
            If a non-zero integer ID, the ID of the object that is our
            destination.
            If a coordinate tuple, the fixed location of our destination.
            If None (or zero), then clear any previous destination and don't
            set a new one.
        """
        old_dest = self.dest
        self.dest = None
        if type(id_or_loc_or_None) is tuple:
            # Parameter is a location
            loc = id_or_loc_or_None
            track_id = None
        elif isinstance(id_or_loc_or_None, SimObj):
            # Parameter is an object
            loc = id_or_loc_or_None.loc
            track_id = id_or_loc_or_None.id
        else:
            # Parameter is an object ID
            sim_obj = self.getObject(id_or_loc_or_None)
            if sim_obj:
                loc = sim_obj.loc
                track_id = sim_obj.id
            else:
                loc = None
                track_id = None
        if loc:
            self.dest = DestinationMarker(self._universe, loc, 
                                          track_id=track_id)
            if self.dock_station:
                self._unDock()
        if old_dest:
            old_dest.delete()

    # Helper methods

    def _turn(self, turn_inc):
        # turn_inc: degrees / sec**2
        self.va += turn_inc
        if self.va < -self.MAX_TURN_RATE:
            self.va = -self.MAX_TURN_RATE
        elif self.va > self.MAX_TURN_RATE:
            self.va = self.MAX_TURN_RATE

    def _accel(self, speed_inc):
        # speed_inc: distance units / sec**2
        vx, vy = self.vel
        vx, vy = movepolar(vx, vy, self.angle, speed_inc)
        speed = math.sqrt(vx*vx + vy*vy)
        if speed  > self.MAX_SPEED:
            ratio = self.MAX_SPEED / speed
            vx *= ratio
            vy *= ratio
        self.vel = vx, vy

    def _dockUpdate(self):
        # Move to docking station
        dock_obj, dock_angle = self.dock_station
        x, y = dock_obj.loc
        x, y = movepolar(x, y, dock_angle, DOCK_DISTANCE)
        self.loc = x, y
        self.angle = dock_angle + 180.0
        self.vel = dock_obj.vel
        self.va = 0.0

    def _unDock(self):
        dock_obj, dock_angle = self.dock_station
        dock_obj.requestUnDock(self)
        self.dock_station = None
        self.postMessage("Undocked")


class StarShip(StarVessel, ContainerObj):
    
    """
    A StarShip is a StarVessel that has a humanoid pilot, can carry cargo,
    and can fire missiles.

    The 'user' attribute is the name of the pilot.
    """

    MISSILE_CAP = 10
    REARM_INTERVAL_TICKS = TICKS_PER_SECOND * 4

    def __init__(self, universe, loc,
                user=None,       # the current pilot
                **init_params):
        self.user = user # must set user first
        super(StarShip, self).__init__(universe, loc, **init_params)
        self._missiles = 10
        self._rearm_timer = self.REARM_INTERVAL_TICKS

    def getView(self):
        """
        Return a tuple containing the ID, the location, and the current
        visible state of the starship. This tuple contains the parameters to
        the update() method on the view object.
        """
        return super(StarShip, self).getView() + (self.user,)

    # StarShip-specific commands

    def cmd_fire(self):
        if self._missiles <= 0:
            self.postMessage("Out of missiles")
            return
        if not self.selection:
            self.postMessage("Select a target first")
            return
        target = self.getObject(self.selection.track_id)
        if not target:
            target = self.selection # target a location, not an object
        a = self.angle * math.pi / 180.0
        vx = self.MISSILE_LAUNCH_SPEED * math.sin(a)
        vy = self.MISSILE_LAUNCH_SPEED * math.cos(a)
        vel = (self.vel[0] + vx, self.vel[1] + vy)
        missile = self._createMissile(self.loc, self.angle, vel)
        missile.cmd_setDestination(target)
        self._missiles -= 1

    def arrivalTrigger(self):
        # Called when the StarShip arrives at its destination.
        dest_obj = self.getObject(self.dest.track_id)
        if dest_obj and isinstance(dest_obj, StarVessel):
            # Continue following if it is (potentially) a moving object
            return
        if dest_obj and isinstance(dest_obj, StarBase):
            self.cmd_dock()
        self.cmd_setDestination(None)
    
    def _createMissile(self, loc, angle, vel):
        return Missile(self._universe, loc, color='blue', angle=angle, 
                       vel=vel, va=0.0)

    def _dockUpdate(self):
        super(StarShip, self)._dockUpdate()
        # Re-arm
        self._rearm_timer -= 1
        if self._rearm_timer <= 0:
            self._rearm_timer = self.REARM_INTERVAL_TICKS
            if self._missiles < self.MISSILE_CAP:
                self._missiles += 1


class Garrett1(StarShip):

    pass


class Nathan1(StarShip):

    pass


class Nathan2(StarShip):

    pass


class Evan1(StarShip):

    pass


class FuonInvader(StarShip):
    
    def _createMissile(self, loc, angle, vel):
        return FuonMissile(self._universe, loc, color='red', angle=angle,
                           vel=vel, va=0.0)


class Drone(StarVessel):

    """
    A Drone is a StarVessel that cannot carry a humanoid pilot, but has a
    simple robot pilot instead. Drones are intended to be expendable.
    """


class Missile(Drone):

    """
    A missile is a Drone that carries a warhead and explodes when it reaches
    its destination.
    """

    def arrivalTrigger(self):
        Explosion(self._universe, self.loc)
        self.delete()


class FuonMissile(Missile):

    pass


class StarGram(Drone):

    """
    A StarGram is a messenger drone. When the StarGram reaches its
    destination it delivers its message contents and then disappears.
    """

    def __init__(self, universe, loc,
                sender=None,       # message sender
                content='Hello',   # message content
                **init_params):
        init_params.pop('color', None) # remove color param if given
        super(StarGram, self).__init__(universe, loc, color='blue', 
                                       **init_params)
        self.sender = sender
        self.content = content

    def arrivalTrigger(self):
        dest = self.getObject(self.dest.track_id)
        if getattr(dest, 'postMessage', None):
            msg = "StarGram: %s" % str(self.content)
            dest.postMessage(msg)
        self.delete()


class Explosion(SimObj):
    """
    A transient energy burst that grows quickly and then disappears.
    """

    active = True

    def __init__(self, universe, loc, 
                 limit=0.1, 
                 rate=0.01,
                 **init_params):
        """
        limit: The final radius of the explosion, in sector units
        rate: The expansion rate of the explosion radius, in units per tick
        """
        super(Explosion, self).__init__(universe, loc, **init_params)
        self._limit = limit
        self._rate =  abs(rate)
        self._cur_radius = 0.0

    def update(self):
        """
        Update the explosion.
        """
        super(Explosion, self).update()
        self._cur_radius += self._rate
        if self._cur_radius > self._limit:
            self.delete()

    def getView(self):
        return super(Explosion, self).getView() + (self._cur_radius,)


# end-of-file
