# starsim/auto.py
# David Handy  August 2004
"""
StarSim auto-pilot
"""

import math

import starsim.const


def angleDiff(angle, loc, dest):
    """
    Calculate the angular difference between the angle parameter and the
    angle of the line segment from (loc) to (dest). The loc and dest
    parameters must be two different coordinate locations.

    angle: A compass-style angle in degrees
    loc: Line segment start coordinates (x1, y1)
    dest: Line segment end coordinates (x2, y2)
    
    Return the angular difference, in degrees, -180 <= diff <= 180.
    If loc == dest, zero is returned.

    >>> diff = angleDiff(45.0, (0.0, 0.0), (10.0, 0.0))
    >>> print "%.1f" % diff
    45.0
    >>> diff = angleDiff(90.0, (0.0, 0.0), (10.0, 10.0))
    >>> print "%.1f" % diff
    -45.0
    """
    # Convert parameters to two unit vectors (x, y) and (xp, yp)
    # Then calculate angle theta by solving the equations:
    #   xp = x * cos(theta) - y * sin(theta)
    #   yp = y * cos(theta) + x * sin(theta)
    a = (angle * math.pi) / 180.
    x, y = math.sin(a), math.cos(a) # compass-style angle interpretation
    x1, y1 = loc
    x2, y2 = dest
    x3, y3 = (x2 - x1, y2 - y1)
    d = math.sqrt(x3*x3 + y3*y3)
    xp, yp = x3/d, y3/d
    cos_theta = xp * x + yp * y
    sin_theta = xp * -y + yp * x
    result_rad = math.atan2(sin_theta, cos_theta)
    return (-result_rad * 180.) / math.pi


class AutoPilot:
    """
    Navigation controller for taking a ship (or drone or missle) to a
    destination.

    Example use:
        ship = a StarShip or drone or missle or something that navigates
        dest = coordinates of destination
        a = AutoPilot(ship)

    """

    CLOSE_ENOUGH = starsim.const.DOCK_DISTANCE - 0.003

    def __init__(self, ship, angle_brake_thresh, boost_brake_thresh):
        """
        ship:
            The ship that the autopilot controls.
            Expect angle and loc attributes.
        angle_brake_thresh:
            Tuning parameter for controlling turning. The angular distance
            in degrees beyond which we use full turn power.
        boost_brake_thresh:
            Tuning parameter for controlling boost. The distance from
            destination beyond which full boost power is advised.
        """
        # check ship parameter for the required attributes
        test1, test2 = ship.angle, ship.loc
        self.ship = ship
        self.angle_brake_thresh = angle_brake_thresh
        self.boost_brake_thresh = boost_brake_thresh

    def advise(self, dest):
        """
        Calculate the auto-pilot's advice about turning and boosting.

        dest: The current desired destination coordinate (x, y)

        Return (recommended_turn_rate, recommended_boost_power)
            recommended_turn_rate: -1.0 to 1.0, inclusive (negative=left)
            recommended_boost_power: -1.0 to 1.0, inclusive (negative=back)

        Return exactly (0.0, 0.0) when we are "close enough."
        """
        angle = self.ship.angle
        loc = self.ship.loc
        deltaX = dest[0] - loc[0]
        deltaY = dest[1] - loc[1]
        dist = math.sqrt(deltaX*deltaX + deltaY*deltaY)
        if dist <= self.CLOSE_ENOUGH:
            return (0.0, 0.0)
        angle_diff = angleDiff(angle, loc, dest)
        if abs(angle_diff) >= self.angle_brake_thresh:
            if angle_diff >= 0.0:
                turn_rate = 1.0
            else:
                turn_rate = -1.0
        else:
            turn_rate = angle_diff / self.angle_brake_thresh
        if dist >= self.boost_brake_thresh:
            boost_power = 1.0
        else:
            boost_power = dist / self.boost_brake_thresh
        boost_power *= math.cos((angle_diff * math.pi) / 180.)
        return (turn_rate, boost_power)


# end-of-file
