# starsim/player/view.py
# David Handy  14 July 2004
"""
View module for the starsim game.

Contains view classes to correspond with the model classes in starsim.model.
"""

import math

from cpif.graphics import Transform2D

PIXELS_PER_SECTOR = 1000.0 # In ScanDisplay mode


class ViewBase(object):
    """
    Base class for all viewable objects.
    """

    # No object should have a bigger pixel radius than this.
    SIZE = 20

    def __init__(self, id, *params):
        """
        id: The persistent, unique ID for this object.
        params: Other static viewable information about this object.
        """
        self.id = id
        self.loc = (0.0, 0.0)        # location in universe coordinates
        self.bbox = [(0, 0), (0, 0)] # bounding box in window coordinates

    def update(self, loc, *params):
        """
        Called when viewable information about this object has changed.

        loc: The new location of this object, in universe coordinates.
        params: Other viewable information, as per this object's specs.
        """
        raise NotImplementedError("Interface method not implemented.")

    def draw(self, dw, pov):
        """
        Called when this object should be (re-)drawn on the window.

        dw: DrawingWindow
        pov: Point Of View, the Transform2D object of the viewpoint.
        """
        raise NotImplementedError("Interface method not implemented.")

    def delete(self, dw):
        """
        Called when this object should be deleted from the window.

        dw: DrawingWindow
        """
        raise NotImplementedError("Interface method not implemented.")


class Unknown(ViewBase):
    """
    Placeholder for an object of unknown type.
    """
    
    def __init__(self, id, class_name=None):
        ViewBase.__init__(self, id)
        if class_name:
            self.class_name = class_name
        else:
            self.class_name = str(id)
        self.loc = (0.0, 0.0)
        self.loc_changed = True
        self.draw_id = None
        self.text_id = None

    def update(self, loc, *params):
        if loc != self.loc:
            self.loc_changed = True
            self.loc = loc

    def draw(self, dw, pov):
        for draw_id in (self.draw_id, self.text_id):
            if draw_id:
                dw.delete(draw_id)
        x, y = pov.transform([self.loc])[0]
        size = 20
        self.draw_id = dw.circle(x, y, size, 'white', fill='yellow')
        w, h = dw.getTextSize(self.class_name)
        tx, ty = x - (w / 2), y - (h / 2)
        self.text_id = dw.drawText(tx, ty, self.class_name, color='gray')
        self.loc_changed = False
        self.bbox = [(x - size, y - size), (x + size, y + size)]

    def delete(self, dw):
        if self.draw_id:
            dw.delete(self.draw_id)
        if self.text_id:
            dw.delete(self.text_id)


class Star(ViewBase):
    """
    A star. Doesn't do much, mostly background ornamentation.
    """
    
    def __init__(self, id):
        ViewBase.__init__(self, id)
        self.loc = (0.0, 0.0)
        self.draw_id = None

    def update(self, loc, *params):
        self.loc = loc

    def draw(self, dw, pov):
        x, y = pov.transform([self.loc])[0]
        size = 1
        if not self.draw_id:
            self.draw_id = dw.box(x-size, y-size, x+size, y+size, 'white',
                                  fill='white')
        else:
            dw.moveTo(self.draw_id, x, y)
        self.bbox = [(x - size, y - size), (x + size, y + size)]

    def delete(self, dw):
        if self.draw_id:
            dw.delete(self.draw_id)
            self.draw_id = None


class Marker(ViewBase):
    """
    Base class for all marker views.
    """

    POINTS = [] # derived class has (x, y) pairs
    POLYGONS = [] # [(i1, i2,...), (i4, i5, ...), ...] i(n) = vertex index
    COLOR = 'green'

    def __init__(self, id, *params):
        """
        id: The persistent, unique ID for this object.
        params: Other static viewable information about this object.
        """
        ViewBase.__init__(self, id)
        # disable pickObject by setting bbox to None
        self.bbox = None
        self.prev_pos = None
        self.draw_id = None

    def update(self, loc, *params):
        """
        Called when viewable information about this object has changed.

        loc: The new location of this object, in universe coordinates.
        params: Other viewable information, as per this object's specs.
        """
        self.loc = loc

    def points(self, pov):
        angle = 0.0
        scale = (1.0/PIXELS_PER_SECTOR) * self.SIZE
        x, y = self.loc
        t = Transform2D()
        t.set(-angle, scale, scale, x, y)
        t.mult(pov)
        return t.transform(self.POINTS)

    def draw(self, dw, pov):
        """
        Called when this object should be (re-)drawn on the window.

        dw: DrawingWindow
        pov: Point Of View, the Transform2D object of the viewpoint.
        """
        points = self.points(pov)
        if not self.prev_pos:
            self._draw(dw, points)
        else:
            self._redraw(dw, points)

    def delete(self, dw):
        """
        Called when this object should be deleted from the window.

        dw: DrawingWindow
        """
        if self.draw_id:
            dw.delete(self.draw_id)
            self.draw_id = None

    def _draw(self, dw, points):
        self.prev_pos = points[0]
        self.draw_id = dw.beginDrawing()
        for index_list in self.POLYGONS:
            poly_pts = [points[i] for i in index_list]
            dw.polygon(poly_pts, self.COLOR)
        dw.endDrawing()

    def _redraw(self, dw, points):
        px, py = self.prev_pos
        x, y = points[0]
        deltaX = x - px
        deltaY = y - py
        self.prev_pos = points[0]
        dw.moveBy(self.draw_id, deltaX, deltaY)


class DestinationMarker(Marker):

    POINTS = [
        (-0.0625, 1.0), (0.0625, 1.0), (0.0, 0.5),
        (0.5, 0.0), (1.0, 0.0625), (1.0, -0.0625),
        (-0.0625, -1.0), (0.0625, -1.0), (0.0, -0.5),
        (-1.0, 0.0625), (-1.0, -0.0625), (-0.5, 0.0),
        ]
    POLYGONS = [(0, 1, 2), (3, 4, 5), (6, 7, 8), (9, 10, 11)]
    COLOR = 'green'


class SelectionMarker(Marker):

    POINTS = [
        (0.5, 1.0), (1.0, 1.0), (1.0, 0.5),
        (1.0, -0.5), (1.0, -1.0), (0.5, -1.0),
        (-0.5, -1.0), (-1.0, -1.0), (-1.0, -0.5),
        (-1.0, 0.5), (-1.0, 1.0), (-0.5, 1.0),
        ]
    POLYGONS = [(0, 1, 2), (3, 4, 5), (6, 7, 8), (9, 10, 11)]
    COLOR = 'green'


class StarShip(ViewBase):
    
    POINTS = [(0., 0.), (0., 1.), (0.5, -1.), (0., -0.5), (-0.5, -1.)]
    CENTER_INDEX = 0

    def __init__(self, id, color):
        ViewBase.__init__(self, id)
        self.color = color
        ###
        self.draw_id = None
        self.text_id = None
        ###
        self.prev_ppos = None # pixel position
        self.prev_angle = None
        self.prev_user = None

    def update(self, loc, angle, user=None):
        self.loc = loc
        self.angle = angle
        self.user = user

    def points(self, pov):
        x, y = self.loc
        scale = (1.0/PIXELS_PER_SECTOR) * self.SIZE
        t = Transform2D()
        t.set(-self.angle, scale, scale, x, y)
        t.mult(pov)
        return t.transform(self.POINTS)

    def draw(self, dw, pov):
        points = self.points(pov)
        ppos = points[self.CENTER_INDEX]
        if self.prev_ppos:
            dx, dy = ppos[0] - self.prev_ppos[0], ppos[1] - self.prev_ppos[1]
        else:
            dx, dy = 0, 0
        if self.text_id:
            if self.prev_user <> self.user:
                dw.delete(self.text_id)
                if self.user:
                    self._label(dw, ppos, self.user)
            else:
                dw.moveBy(self.text_id, dx, dy)
        elif self.user:
            self._label(dw, ppos, self.user)
        if not self.draw_id:
            self._draw(dw, points)
        elif self.prev_angle <> self.angle:
            self._redraw(dw, points)
        else:
            dw.moveBy(self.draw_id, dx, dy)
        self.prev_ppos = ppos
        self.prev_angle = self.angle
        self.prev_user = self.user
        # protocol required by pickObject()
        xcoords, ycoords = zip(*points)
        self.bbox = [(min(xcoords), min(ycoords)), 
                     (max(xcoords), max(ycoords))]

    def delete(self, dw):
        if self.draw_id:
            dw.delete(self.draw_id)
        if self.text_id:
            dw.delete(self.text_id)

    def _draw(self, dw, points):
        self.draw_id = dw.polygon(points[1:], 'white', self.color)

    def _redraw(self, dw, points):
        dw.delete(self.draw_id)
        self._draw(dw, points)

    def _label(self, dw, ppos, text):
        size = self.SIZE
        x, y = ppos
        tw, th = dw.getTextSize(text)
        self.text_id = dw.drawText(x - (tw/2), y - size - th, text, 'white')


class Garrett1(StarShip):
    """
    View of Garrett's model 1 starship design.
    """
    
    POINTS = [
             (0., 1.),
             (0.25, 0.5),
             (0.5, 0.5),
             (0.5, 0.25),
             (1., -0.5),
             (0.25, -0.5),
             (0., -1.),
             (-0.25, -0.5),
             (-1., -0.5),
             (-0.5, 0.25),
             ###
             (-0.5, 0.5),
             (-0.25, 0.5),
             (-0.125, 0.75),
             (0.125, 0.75),
             (-0.875, -0.75),
             (-0.625, -0.75),
             (-0.5, -0.5),
             (0.5, -0.5),
             (0.625, -0.75),
             (0.875, -0.75),
             ###
             (0.25, 0.25),
             (0.25, -0.25),
             (-0.25, -0.25),
             (-0.25, 0.25),
             (-0.875, -0.5),
             (-0.625, -0.5),
             (0.625, -0.5),
             (0.875, -0.5),
             (0., 0.),
             ]
    CENTER_INDEX = 28

    def __init__(self, id, color):
        StarShip.__init__(self, id, color)

    def _draw(self, dw, points):
        self.draw_id = dw.beginDrawing()
        dw.polygon(points[:12], 'white', self.color)
        x1, y1 = points[12]; x2, y2 = points[13]
        dw.line(x1, y1, x2, y2, 'white', width=2)
        corners = [points[2], points[17], points[16], points[10]]
        dw.polygon(corners, 'white')
        dw.polygon(points[20:24], 'white')
        x1, y1 = points[24]; x2, y2 = points[14]
        dw.line(x1, y1, x2, y2, 'white')
        x1, y1 = points[25]; x2, y2 = points[15]
        dw.line(x1, y1, x2, y2, 'white')
        x1, y1 = points[26]; x2, y2 = points[18]
        dw.line(x1, y1, x2, y2, 'white')
        x1, y1 = points[27]; x2, y2 = points[19]
        dw.line(x1, y1, x2, y2, 'white')
        dw.endDrawing()


class Nathan1(StarShip):
    """
    View of Nathan's model 1 design. Looks like an N on its side.
    """

    POINTS = [ ( 0.0,  0.0),
               ( 0.0,  1.0),
               (-0.4,  0.4),
               ( 0.4,  0.4),
               (-0.5, -0.5),
               ( 0.5, -0.5),
               (-0.4, -0.5),
               (-0.2, -0.7),
               ( 0.0, -0.5),
               ( 0.2, -0.7),
               ( 0.4, -0.5),
               ]

    def __init__(self, id, color):
        StarShip.__init__(self, id, color)

    def _draw(self, dw, points):
        self.draw_id = dw.beginDrawing()
        dw.polygon(points[1:4], 'white', fill=self.color)
        dw.lines(points[3:6], 'white', width=2)
        dw.polygon(points[6:9], 'white', fill='white')
        dw.polygon(points[8:11], 'white', fill='white')
        dw.endDrawing()


class Nathan2(StarShip):
    """
    View of Nathan's model 2 design.
    """

    POINTS = [ ( 0.0,  0.0),
               ( 0.0,  1.0),
               ( 0.5,  0.1),
               ( 0.8,  0.4),
               ( 0.9, -0.3),
               ( 0.6, -0.1),
               ( 1.0, -0.7),
               (-1.0, -0.7),
               (-0.6, -0.1),
               (-0.9, -0.3),
               (-0.8,  0.4),
               (-0.5,  0.1),
               ( 0.3,  0.3),
               ( 0.3, -0.3),
               (-0.3, -0.3),
               (-0.3,  0.3),
               ( 0.15, 0.15),
               ( 0.15,-0.15),
               (-0.15,-0.15),
               (-0.15, 0.15),
               ( 0.7, -0.7),
               ( 0.5, -1.0),
               ( 0.3, -0.7),
               (-0.3, -0.7),
               (-0.5, -1.0),
               (-0.7, -0.7),
               ]

    def __init__(self, id, color):
        StarShip.__init__(self, id, color)

    def _draw(self, dw, points):
        self.draw_id = dw.beginDrawing()
        dw.polygon(points[1:12], 'white', fill=self.color)
        dw.polygon(points[12:16], 'white')
        dw.polygon(points[16:20], 'white')
        dw.polygon(points[20:23], 'red', fill='orange')
        dw.polygon(points[23:26], 'red', fill='orange')
        dw.endDrawing()


class Evan1(StarShip):
    """
    View of Evan's model 1 design.
    """

    POINTS = [ ( 0.0,  0.0),
               ( 0.0,  1.0),
               ( 0.7,  0.7),
               ( 0.8,  0.0),
               ( 1.0, -0.1),
               ( 1.0, -0.4),
               ( 0.8, -0.5),
               ( 0.7, -0.7),
               ( 0.5, -0.7),
               ( 0.3, -1.0),
               ( 0.0, -0.7),
               (-0.3, -1.0),
               (-0.5, -0.7),
               (-0.8, -0.7),
               (-0.8, -0.5),
               (-1.0, -0.4),
               (-1.0, -0.1),
               (-0.8,  0.0),
               (-0.7,  0.7),
               ( 0.0,  0.8),
               ( 0.4,  0.4),
               (-0.4,  0.4),
               ( 0.05, 0.65),
               ( 0.25, 0.45),
               ( 0.05, 0.45),
               (-0.05, 0.65),
               (-0.05, 0.45),
               (-0.25, 0.45),
               ]

    def __init__(self, id, color):
        StarShip.__init__(self, id, color)
        self.polygons = [] # [(id, (start, end))]

    def _draw(self, dw, points):
        self.draw_id = dw.beginDrawing()
        p1 = dw.polygon(points[1:19], 'white', fill=self.color)
        p2 = dw.polygon(points[19:22], 'white')
        p3 = dw.polygon(points[22:25], 'white')
        p4 = dw.polygon(points[25:28], 'white')
        dw.endDrawing()
        self.polygons = [(p1, (1,19)), (p2, (19,22)), (p3, (22,25)), 
                         (p4, (25,28))]

    def _redraw(self, dw, points):
        # change the vertex coordinates of the polygons
        for p, (start, end) in self.polygons:
            dw.setCoords(p, points[start:end])


class FuonInvader(StarShip):

    POINTS = [ ( 0.0,  0.0),
               ( 0.3,  0.6),
               ( 0.1,  0.4),
               ( 0.2,  0.1),
               ( 0.9,  0.2),
               ( 0.9, -0.2),
               ( 0.2, -0.1),
               ( 0.4, -0.5),
               (-0.4, -0.5),
               (-0.2, -0.1),
               (-0.9, -0.2),
               (-0.9,  0.2),
               (-0.2,  0.1),
               (-0.1,  0.4),
               (-0.3,  0.6),
               ]

    def __init__(self, id, color):
        super(FuonInvader, self).__init__(id, color)

    def _draw(self, dw, points):
        self.draw_id = dw.beginDrawing()
        dw.polygon(points[2:14], 'white', fill=self.color)
        dw.lines(points[1:3], 'blue', width=2)
        dw.lines(points[13:15], 'blue', width=2)
        dw.endDrawing()


class StarBase(StarShip):

    POINTS = [ ( 0.0,  0.0),
               ( 0.0,  0.9),
               ( 0.4,  0.0),
               ( 0.0, -0.4),
               (-0.4,  0.0),
               ]

    def __init__(self, id, color):
        super(StarBase, self).__init__(id, color)

    def _draw(self, dw, points):
        x0, y0 = points[0]
        x1, y1 = points[1]
        dx = x1 - x0
        dy = y1 - y0
        r = math.sqrt(dx * dx + dy * dy)
        self.draw_id = dw.beginDrawing()
        dw.circle(x0, y0, r, 'white', fill=self.color)
        dw.polygon(points[1:], 'white', fill='', width=2)
        dw.endDrawing()


class FuonBase(StarBase):

    POINTS = [ ( 0.0,  0.0),
               ( 0.5,  0.5),
               ( 0.9, -0.7),
               (-0.9, -0.7),
               (-0.5,  0.5),
               (-0.5, -0.5),
               ( 0.5, -0.5),
               ]

    def __init__(self, id, color):
        super(FuonBase, self).__init__(id, color)

    def _draw(self, dw, points):
        self.draw_id = dw.polygon(points[1:], 'white', fill=self.color)


class Missile(StarShip):

    POINTS = [ ( 0.0,  0.0),
               ( 0.0,  0.2),
               ( 0.8,  0.8),
               ( 0.2,  0.0),
               ( 0.8, -0.8),
               ( 0.0, -0.2),
               (-0.8, -0.8),
               (-0.2,  0.0),
               (-0.8,  0.8),
               ]

    def __init__(self, id, color):
        super(Missile, self).__init__(id, color)

    def _draw(self, dw, points):
        self.draw_id = dw.polygon(points[1:], 'white', fill=self.color, 
                                  width=2)


class StarGram(StarShip):

    POINTS = [ ( 0.0,  0.0),
               ( 0.0,  0.4),
               ( 0.4,  0.0),
               ( 0.0, -0.2),
               (-0.4,  0.0),
               ]

    def __init__(self, id, color='blue'):
        super(StarGram, self).__init__(id, color)

    def _draw(self, dw, points):
        self.draw_id = dw.polygon(points[1:], 'white', fill=self.color, 
                                  width=1)


class FuonStarGram(StarGram):

    def __init__(self, id):
        super(StarGram, self).__init__(id, 'red')


class Explosion(ViewBase):
    """
    A transient energy burst that grows quickly and then disappears.
    """
    
    def __init__(self, id):
        super(Explosion, self).__init__(id)
        self.loc = (0.0, 0.0)
        self.draw_id = None

    def update(self, loc, radius):
        self.loc = loc
        self.radius = radius

    def draw(self, dw, pov):
        x, y = pov.transform([self.loc])[0]
        r = self.radius * PIXELS_PER_SECTOR
        self.bbox = [(x - r, y - r), (x + r, y + r)]
        if not self.draw_id:
            self.draw_id = dw.circle(x, y, r, 'white', fill='white')
        else:
            try:
                dw.setCoords(self.draw_id, self.bbox)
            except:
                print self.bbox
                raise

    def delete(self, dw):
        if self.draw_id:
            dw.delete(self.draw_id)
            self.draw_id = None


# end-of-file
