# starsim/player/tiles.py
# David Handy  19 August 2005
"""
UI components made of non-overlapping canvas areas called "tiles".
"""

import logging
import math
import sets
import sys

from cpif.graphics import DrawingWindow
from cpif.graphics import Transform2D
from cpif.rpc import Disconnected

from starsim.player import view

logger = logging.getLogger('starsim.player.tiles')


class BaseTile(object):
    """
    Base class for non-overlapping graphical "tiles" to control portions of
    the user display/interface.
    """

    def __init__(self, player, pos, size, id=None, **kwparams):
        """
        player -- 
            Root "player" object that provides the following attributes
            and methods:
            player.dw # the main DrawingWindow
            player.subscribe(event, callback, expire_after_count)
            player.unsubscribe(event)
            player.notify(event, data) # called from server
            player.callMethod(method, params, callback) # call server
            player.disconnect() # call on Disconnected exception
            player.ship_id # Current ship user controls
        pos -- (x, y) position of upper left corner of tile
        size -- (width, height) size of tile in pixels
        id (optional) -- The Universe object ID representing the server-side
                         entity that goes with this tile, if any.
        kwparams -- initialization parameters used by derived classes
        """
        self.player = player
        self.pos = pos
        self.size = size
        self.id = id
        self.__children = sets.Set()

    def addChild(self, child_tile):
        """Add a tile to the managed list of child tiles."""
        self.__children.add(child_tile)

    def deleteChild(self, child_tile):
        """Remove the tile from the set of child tiles and destroy it."""
        self.__children.discard(child_tile)
        child_tile.destroy()

    def destroy(self):
        """
        Erase the items drawn in the tile.

        The BaseTile implementation of this method calls destroy() on all of
        the child tiles. Derived classes should call the base class
        destroy() and then delete its own drawn items.
        """
        for child_tile in self.__children:
            child_tile.destroy()
        self.__children = sets.Set()
        return False

    def getChildTileAtPos(self, x, y):
        """
        Return the first child tile that the coordinates (x, y) falls inside
        of, or None if no matching tile is found.
        """
        for child_tile in self.__children:
            c_x, c_y = child_tile.pos
            c_w, c_h = child_tile.size
            if ((x >= c_x and x < (c_x + c_w)) and
                (y >= c_y and y < (c_y + c_h))):
                return child_tile
        return None

    def onClick(self, mouse_event):
        """
        This method is called when a mouse button is pressed over 
        the tile.

        mouse_event is an object with the following attributes:
            button_num:
                1 for left button, 2 for middle, 3 for right,
                0 for no button
            x: the X window coordinate of the mouse event
            y: the Y window coordinate of the mouse event

        The BaseTile implementation of this method redirects this event
        to the first child tile that contains the mouse coordinates. Derived
        classes usually either do something with mouse_event or call the base
        class onClick(mouse_event), but not both.
        """
        child_tile = self.getChildTileAtPos(mouse_event.x, mouse_event.y)
        if child_tile:
            return child_tile.onClick(mouse_event)

    def onKeyDown(self, key_event):
        """
        This method is called when a key is pressed, and the parent child has
        decided to send keyboard events to this tile.
        """
        pass

    def redraw(self, pos, size):
        """
        (Re)draw the tile at a new position and size.

        Call the BaseTile version of this method to set the new size and
        position of the tile.
        """
        self.pos = pos
        self.size = size


class BorderedTile(BaseTile):
    """Tile with a border."""

    def __init__(self, player, pos, size, border_color='yellow',
                 **kwparams):
        super(BorderedTile, self).__init__(player, pos, size, **kwparams)
        x, y = pos
        w, h = size
        self.__border = self.player.dw.box(x, y, x+w-1, y+h-1, border_color)

    def destroy(self):
        super(BorderedTile, self).destroy()
        if self.__border:
            dw = self.player.dw
            dw.delete(self.__border)
            self.__border = None

    def redraw(self, pos, size):
        super(BorderedTile, self).redraw(pos, size)
        endp = (pos[0] + size[0] - 1, pos[1] + size[1] - 1)
        self.player.dw.setCoords(self.__border, [pos, endp])


class LabeledTile(BaseTile):
    """Tile with a label at the top."""

    padx = 2  # pixels. TODO: make this configurable
    pady = 2  # pixels. TODO: make this configurable

    def __init__(self, player, pos, size, label_text='', label_color='white',
                 **kwparams):
        super(LabeledTile, self).__init__(player, pos, size, **kwparams)
        dw = player.dw
        x, y = pos
        label_x = x + self.padx
        label_y = y + self.pady
        label_w, label_h = dw.getTextSize(label_text)
        self.label_text = label_text
        self.label_x = label_x
        self.label_y = label_y
        self.label_w = label_w
        self.label_h = label_h
        dw = player.dw
        self.__label_id = dw.drawText(label_x, label_y, label_text, label_color)

    def destroy(self):
        super(LabeledTile, self).destroy()
        if self.__label_id:
            self.player.dw.delete(self.__label_id)
            self.__label_id = None


class ButtonTile(BaseTile):

    """Button that can be pressed to call a function."""

    def __init__(self, player, pos, size, button_color='white',
                 button_callback=None, **kwparams):
        """
        button_color -- color of interior of button
        button_callback -- callable object (i.e. function) that will be
        called like this:
            button_callback(mouse_event)
        """
        super(ButtonTile, self).__init__(player, pos, size, **kwparams)
        dw = self.player.dw
        x, y = pos
        w, h = size
        padx = int(w * 0.05)
        pady = int(h * 0.05)
        x1 = x + padx
        x2 = x + w - padx
        y1 = y + pady
        y2 = y + h - pady
        self.__button = dw.oval(x1, y1, x2, y2, button_color, 
                                fill=button_color)
        if button_callback:
            self.onButtonPressCall(button_callback)

    def destroy(self):
        super(ButtonTile, self).destroy()
        if self.__button:
            self.onButtonPressCall(None)
            self.player.dw.delete(self.__button)
            self.__button = None

    def onButtonPressCall(self, callback):
        if self.__button:
            dw = self.player.dw
            dw.onClickCall(self.__button, callback)


class LabeledButtonTile(BaseTile):
    """
    A button with a label above or to the side.
    """
    # Uses composition rather than inheritance

    def __init__(self, player, pos, size, 
        label_text='', label_color='white', label_position='above', 
        **kwparams):
        """
        Initialize labeled button.
        label_position --
            'above' for label above button
            'right' for label to right of button
        """
        super(LabeledButtonTile, self).__init__(player, pos, size, **kwparams)
        dw = player.dw
        x, y = pos
        w, h = size
        label_w, label_h = dw.getTextSize(label_text)
        label_w += 2 * LabeledTile.padx
        label_h += 2 * LabeledTile.pady
        label_size = (label_w, label_h)
        if label_position == 'above':
            label_pos = pos
            button_side = h - label_h
            button_size = (button_side, button_side) # square, under label
            button_pos = (x + (w//2) - (button_side//2), y + label_h)
        elif label_position == 'right':
            button_pos = pos
            button_size = (h, h) # square, fits height of tile
            label_pos = (x + button_size[0], y + (h//2) - (label_h//2))
        else:
            raise ValueError("Invalid label_position: " + repr(label_position))
        button_tile = ButtonTile(player, button_pos, button_size, **kwparams)
        self.addChild(button_tile)
        self.button_tile = button_tile
        label_tile = LabeledTile(player, label_pos, label_size,
                                 label_text=label_text,
                                 label_color=label_color,
                                 **kwparams)
        self.addChild(label_tile)

    def destroy(self):
        super(LabeledButtonTile, self).destroy()
        self.button_tile = None

    # Add methods to support the ButtonTile interface

    def onButtonPressCall(self, callback):
        if self.button_tile:
            self.button_tile.onButtonPressCall(callback)


class BorderedLabeledButtonTile(BorderedTile, LabeledButtonTile):
    pass


class HighlightableTile(BaseTile):
    """A tile that supports the highlight() method to indicate whether it is
    selected/deselected. Initially the tile is not highlighted."""

    def __init__(self, player, pos, size,
                 highlight_width=2,
                 highlight_color='white',
                 **kwparams):
        super(HighlightableTile, self).__init__(player, pos, size, **kwparams)
        self.highlight_width = highlight_width
        self.highlight_color = highlight_color
        self.__highlight_id = None

    def destroy(self):
        super(HighlightableTile, self).destroy()
        if self.__highlight_id:
            dw = self.player.dw
            dw.delete(self.__highlight_id)
            self.__highlight_id = None

    def highlight(self, turn_on=True):
        """Turn the highlighting on or off depending on the turn_on
        parameter, which is True (on) by default."""
        if turn_on and self.__highlight_id:
            # already on
            return
        if not turn_on and not self.__highlight_id:
            # already off
            return
        dw = self.player.dw
        x, y = self.pos
        w, h = self.size
        if turn_on:
            self.__highlight_id = dw.box(x, y, x+w, y+h, self.highlight_color,
                                         width=self.highlight_width)
        else:
            dw.delete(self.__highlight_id)
            self.__highlight_id = None


class CargoManagerTile(BorderedTile):

    """
    The CargoManager has a left section for the StarBase hold, and
    a right section for the StarShip hold, each with a descriptive label
    at the top. There is a bottom section for displaying the description
    of objects when you click on them.
    """
    
    def __init__(self, player, pos, size, border_color='green', **kwparams):
        super(CargoManagerTile, self).__init__(player, pos, size,
                                               border_color=border_color,
                                               **kwparams)
        dw = player.dw
        x, y = pos
        w, h = size
        left_pos = (x, y)
        right_pos = (x + (w//2), y)
        hold_size = (w//2, h)
        self.__left_hold = CargoHoldTile(player, left_pos, hold_size,
                                         label_text='StarBase cargo hold',
                                         border_color=border_color,
                                         cargo_manager=self,
                                         **kwparams)
        self.__right_hold = CargoHoldTile(player, right_pos, hold_size,
                                         id=player.ship_id,
                                         label_text='StarShip cargo hold',
                                         border_color=border_color,
                                         cargo_manager=self,
                                         **kwparams)
        self.addChild(self.__left_hold)
        self.addChild(self.__right_hold)
        self.__selected_cargo = None
        self.__drag_icon = None
        player.callMethod('getDockStationId', (), self._dockStationIdReply)

    def destroy(self):
        super(CargoManagerTile, self).destroy()

    def _dockStationIdReply(self, result):
        try:
            dock_station_id = result.getResult()
        except Disconnected:
            self.player.disconnect()
            return
        self.__left_hold.id = dock_station_id
        self.__left_hold.refresh()

    def _selectCargo(self, cargo_tile):
        # Called by a cargo item when it is clicked.
        # This sets up drag-and-drop to the other cargo hold.
        if self.__selected_cargo:
            self.__selected_cargo.highlight(False)
            if cargo_tile is self.__selected_cargo:
                # Clickon on same cargo item again unselects it
                cargo_tile = None
        self.__selected_cargo = cargo_tile
        if cargo_tile:
            cargo_tile.highlight()
            dw = self.player.dw
            dw.onMouseMoveCall(self._dragCargo)
            dw.onMouseUpCall(self._dropCargo)

    def _dragCargo(self, mouse_event):
        dw = self.player.dw
        if self.__drag_icon:
            dw.moveTo(self.__drag_icon, mouse_event.x, mouse_event.y)
        else:
            if self.__selected_cargo:
                cargo = self.__selected_cargo
                x, y = cargo.pos
                w, h = cargo.size
                box_width = cargo.highlight_width
                box_color = cargo.highlight_color
                self.__drag_icon = dw.box(x, y, x+w, y+h, box_color,
                                          width=box_width)

    def _dropCargo(self, mouse_event):
        dw = self.player.dw
        dw.onMouseMoveCall(None)
        dw.onMouseUpCall(None)
        if self.__drag_icon:
            dw.delete(self.__drag_icon)
            self.__drag_icon = None
        if not self.__selected_cargo:
            return
        source = self.__selected_cargo.cargo_hold
        x, y = mouse_event.x, mouse_event.y
        target = self.getChildTileAtPos(x, y)
        if source is target:
            # Source cargo hold is same as target cargo hold -- do nothing
            return
        if not isinstance(target, CargoHoldTile):
            # Target is not a cargo hold -- do nothing
            return
        # Transfer cargo from source to target by calling a remote method
        self.player.callMethod('transferCargo', 
                (self.__selected_cargo.id, source.id, target.id), 
                self._transferCargoReply)

    def _transferCargoReply(self, result):
        try:
            status, msg = result.getResult()
        except Disconnected, e:
            self.player.disconnect(e)
        logger.info("Transfer cargo reply: %s", msg)
        self.__left_hold.refresh()
        self.__right_hold.refresh()


class LabeledBorderedTile(LabeledTile, BorderedTile):
    pass


class CargoHoldTile(LabeledBorderedTile):

    """
    The CargoHoldTile has a label at the top explaining whether it is the
    display of the cargo hold of the StarBase with which you are docked, or
    the cargo hold of your own StarShip. In main area there are icons for each
    object in the cargo hold. When you click on an item, a selection box is
    drawn around it, and the parent cargo manager is notified.

    TODO: Vertical scroll bar for when there are more items than can appear on
    the screen at one time.
    """

    def __init__(self, player, pos, size, 
                 cargo_manager=None, 
                 container_id=None,
                 label_text='',
                 border_color='green',
                 **kwparams):
        super(CargoHoldTile, self).__init__(player, pos, size, 
                                            label_text=label_text,
                                            border_color=border_color,
                                            **kwparams)
        self.__cargo_manager = cargo_manager
        # Calculate standard cargo cell size
        dw = player.dw
        line_size = dw.getTextSize("AAAAAAAAAA 999")
        self.cargo_cell_size = (line_size[0], line_size[1]*3)
        self.cargo_items = []
        self.__cur_id = 0
        # If we know who we are, then refresh cargo contents
        if self.id:
            self.refresh()

    def addCargoItem(self, cargo_info):
        x, y = self.pos
        w, h = self.cargo_cell_size
        next_y = self.label_y + self.label_h + (len(self.cargo_items) * h)
        self.__cur_id += 1
        cargo_id, name, amount = cargo_info[:3]
        cargo_label = "%-10.10s %3d" % (name, amount)
        cargo = CargoTile(self.player, (x, next_y), (w, h),
                          id=cargo_id,
                          cargo_hold=self,
                          label_text=cargo_label,
                          button_color='blue')
        callback = self._makeCargoCallback(cargo)
        cargo.onButtonPressCall(callback)
        self.cargo_items.append(cargo)
        self.addChild(cargo)

    def destroy(self):
        super(CargoHoldTile, self).destroy()

    def refresh(self):
        cargo_items = self.cargo_items
        self.cargo_items = []
        for cargo_item in cargo_items:
            self.deleteChild(cargo_item)
        self.player.callMethod('getCargoList', (self.id,), self._cargoListReply)

    def _cargoListReply(self, result):
        try:
            cargo_info_list = result.getResult()
        except Disconnected:
            return self.player.disconnect()
        for cargo_info in cargo_info_list:
            self.addCargoItem(cargo_info)

    def _makeCargoCallback(self, cargo):
        cargo_manger = self.__cargo_manager
        def callback(mouse_event):
            cargo_manger._selectCargo(cargo)
        return callback


class CargoTile(LabeledButtonTile, HighlightableTile):

    """Display a cargo item in a cargo hold."""

    def __init__(self, player, pos, size, cargo_hold=None, **kwparams):
        super(CargoTile, self).__init__(player, pos, size,
                label_position='right', **kwparams)
        self.cargo_hold = cargo_hold


class ScannerTile(LabeledBorderedTile):
    
    MAX_SCANNER_BACKLOG = 5

    """Implements the short range scanner display."""

    def __init__(self, player, pos, size, 
                 border_color='green', label_text='Short-range scanner',
                 ship_id=None,
                 **kwparams):
        super(ScannerTile, self).__init__(player, pos, size,
                border_color=border_color, label_text=label_text,
                **kwparams)
        ###
        self.width = size[0]
        self.height = size[1]
        self.cx = pos[0] + (size[0]//2)
        self.cy = pos[1] + (size[1]//2)
        self.ship_id = ship_id
        self.objects = {}
        self.time_id = None
        self.__objects_to_delete = []
        ###
        player.subscribe('displayList', self.displayListEvent, 
                         self.MAX_SCANNER_BACKLOG)
        if not self.ship_id:
            player.callMethod('getId', (), self.getIdReply)

    def commandReply(self, result):
        """Generic command reply handler."""
        # Called in response to various rpc requests
        try:
            result.getResult()
        except Disconnected, e:
            self.disconnect(e)

    def destroy(self):
        super(ScannerTile, self).destroy()
        # delete objects that were being displayed
        dw = self.player.dw
        for display_obj in self.objects.itervalues():
            display_obj.delete(dw)
        self.objects = {}
        # delete other leftover objects
        for obj in self._getObjectsToDelete():
            obj.delete(dw)
        # delete status line
        if self.time_id:
            dw.delete(self.time_id)
        # stop display updates
        self.player.unsubscribe('displayList')

    def displayListEvent(self, data):
        """
        Update the display with the latest display items.
        Request information about items in display_list that were unknown.
        """
        self.player.subscribe('displayList', self.displayListEvent, 
                              self.MAX_SCANNER_BACKLOG)
        dw = self.player.dw
        ticks, display_list = data[:2]
        unknown_ids = []
        display_objs = []
        objects = self.objects.copy()
        for items in display_list:
            id = items[0]
            stuff = items[1:]
            display_obj = objects.pop(id, None)
            if not display_obj:
                unknown_ids.append(id)
            else:
                display_obj.update(*stuff)
                display_objs.append(display_obj)
        # delete objects that are no longer being displayed
        for id in objects.iterkeys():
            display_obj = self.objects.pop(id)
            display_obj.delete(dw)
        del objects
        # delete other leftover objects
        for obj in self._getObjectsToDelete():
            obj.delete(dw)
        if self.ship_id:
            ship = self.objects.get(self.ship_id)
            if ship:
                loc = ship.loc
                self._drawObjects(dw, loc, display_objs)
                status = "%9d (%+6.2f, %+6.2f)" % (ticks, loc[0], loc[1])
            else:
                status = "%9d (?, ?)" % ticks
        else:
            status = "%9d" % (ticks,)
        h = dw.getTextSize(status)[1]
        x = self.cx - (self.width//2)
        y = self.cy + (self.height//2) - h
        if self.time_id:
            dw.delete(self.time_id)
        self.time_id = dw.drawText(x, y, status, color='white')
        if unknown_ids:
            self.player.callMethod('queryInitInfo', (unknown_ids,), 
                                   self.queryInitInfoReply)

    def getIdReply(self, result):
        try:
            self.ship_id = result.getResult()
        except Disconnected, e:
            return self.player.disconnect(e)

    def onClick(self, mouse_event):
        obj_id, loc = self.pickObject(mouse_event.x, mouse_event.y)
        if loc:
            x, y = loc
            logger.debug("Clicked on: %s at (%+6.2f, %+6.2f)", obj_id, x, y)
        if mouse_event.button_num == 3: # right button
            command = 'setSelection'
        else: # middle or left button
            command = 'setDestination'
        if self.ship_id:
            if obj_id is not None:
                param = obj_id
            else:
                param = loc
            self.player.callMethod(command, (param,), self.commandReply)

    def pickObject(self, x, y):
        """
        x, y: Pixel location in window coordinates

        Return (pick_id, (ux, uy))
        pick_id:
            The ID of the object at pixel location (x, y), or None if there
            is no such object. If (x, y) is on or near multiple objects,
            return the ID of the object whose center is nearest to (x, y).
        (ux, uy):
            The universe coordinates of the mouse click location
        """
        matches = []
        for display_obj in self.objects.itervalues():
            bbox = getattr(display_obj, 'bbox', None)
            if not bbox:
                continue
            (minx, miny), (maxx, maxy) = display_obj.bbox
            if (minx <= x and x <= maxx and miny <= y and y <= maxy):
                cx = (minx + maxx) / 2.
                cy = (miny + maxy) /2.
                matches.append((display_obj.id, cx, cy))
        result_id = None
        min_d = None
        for id, cx, cy in matches:
            d = math.sqrt(cx*cx + cy*cy)
            if min_d is None or d < min_d:
                min_d = d
                result_id = id
        if result_id:
            result_obj = self.objects.get(result_id)
            loc = result_obj.loc
        else:
            ship = self.objects.get(self.ship_id)
            if ship:
                # Transform pixel coordinates to universe coordinate
                t = Transform2D()
                angle = 0.0
                scale = 1.0 / view.PIXELS_PER_SECTOR
                t.set(-angle, scale, -scale, ship.loc[0], ship.loc[1])
                px, py = (x - self.cx, y - self.cy)
                loc = t.transform([(px, py)])[0]
            else:
                # Can't transform to universe coordinates if we don't know
                # where in the universe we are located.
                loc = None
        return result_id, loc

    def queryInitInfoReply(self, result):
        # Handle reply to RPC request
        try:
            init_list = result.getResult()
        except Disconnected, e:
            return self.player.disconnect(e)
        for items in init_list:
            if not items:
                continue
            class_name, id = items[:2]
            class_ = getattr(view, class_name, None)
            if class_ and not issubclass(class_, view.ViewBase):
                logger.warn("Security violation: attempt to call %r", class_)
                class_ = None
            if not class_:
                display_obj = view.Unknown(id, class_name)
            else:
                try:
                    display_obj = class_(id, *items[2:])
                except:
                    logger.exception("Creating View: %r, %r, %r", 
                                      class_, id, items[2:])
                    display_obj = view.Unknown(id, class_name)
            old_obj = self.objects.pop(id, None)
            if old_obj:
                self._deleteObjectLater(old_obj)
            self.objects[id] = display_obj

    def _deleteObjectLater(self, obj):
        self.__objects_to_delete.append(obj)

    def _getObjectsToDelete(self):
        result = self.__objects_to_delete
        self.__objects_to_delete = []
        return result

    def _drawObjects(self, dw, loc, display_objs):
        pov = self._getPointOfView(loc)
        for display_obj in display_objs:
            display_obj.draw(dw, pov)

    def _getPointOfView(self, loc):
        pov = Transform2D()
        pov.translate(-loc[0], -loc[1])
        pov.scale(view.PIXELS_PER_SECTOR, -view.PIXELS_PER_SECTOR)
        pov.translate(self.cx, self.cy)
        return pov


class PlayerTile(BorderedTile):
    def __init__(self, player, pos, size, **kwparams):
        super(PlayerTile, self).__init__(player, pos, size, **kwparams)
        self.main_tile = None
        self.redraw(pos, size)

    def onKeyDown(self, key_event):
        """Return True iff this control "eats" the event."""
        if key_event.keysym == 'F7':
            # toggle display mode
            if isinstance(self.main_tile, CargoManagerTile):
                self.setMainTile(ScannerTile)
            else:
                self.setMainTile(CargoManagerTile)
            return True
        return super(PlayerTile, self).onKeyDown(key_event)

    ## PlayerTile specific methods

    def redraw(self, pos, size):
        x, y = pos
        w, h = size
        main_side = min(w, h)
        main_size = (main_side, main_side) # square
        main_pos = (x + (w//2) - (main_side//2), y + (h//2) - (main_side//2))
        if not self.main_tile:
            # First time draw - don't call base class redraw
            main_tile_class = BorderedTile
        else:
            super(PlayerTile, self).redraw(pos, size)
            main_tile_class = self.main_tile.__class__
        self.setMainTile(main_tile_class, main_pos, main_size)

    def setMainTile(self, main_tile_class, pos=None, size=None):
        assert main_tile_class
        if self.main_tile:
            self.deleteChild(self.main_tile)
            if not pos:
                pos = self.main_tile.pos
            if not size:
                size = self.main_tile.size
        else:
            assert pos
            assert size
        self.main_tile = main_tile_class(self.player, pos, size)
        self.addChild(self.main_tile)


class HelloTile(BaseTile):

    """Displays the initial hello message and start button."""

    def __init__(self, player, pos, size, **kwparams):
        # XXX under construction
        super(HelloTile, self).__init__(player, pos, size, **kwparams)
        self.__message = LabeledTile(player, label_pos, label_size, 
                                     label_text='Welcome to StarSim!')
        self.__button = LabeledButtonTile(player, button_pos, button_size,
                                          label_text='Click here to begin',
                                          button_callback=self._onButtonPress)

    def _onButtonPress(self, mouse_event):
        pass


# end-of-file
