# playermod.py
# David Handy  17 Oct 2004
"""
starsim player module
"""

import sys
import webbrowser

from cpif.graphics import DrawingWindow
from cpif.rpc import Disconnected, RemoteException, TimeoutException
from cpif.rpc import Connector, getInProcConnection, InProcNode
from cpif.rpc import lookupInetHostInBackground

from starsim import debug
from starsim.const import TIME_PERIOD
from starsim.player import display, view
from starsim import util


class PlayerMessageHandler:
    """
    This is the RPC handler class for responding to messages from the
    server.
    """

    def __init__(self, connection, player):
        """
        connection: an rpc Connection object
        player: a Player class
        """
        self._connection = connection
        self._player = player
        ###
        self._proxy = connection.getProxy(callback=None)
        self._updates = 0
        self._connect_time = util.gettime()

    def update(self, data):
        self._updates += 1
        self._player.update(data)

    def showMessages(self, msgs):
        self._player.showMessages(msgs)

    def stop(self):
        self._player.stop()

    def _printStats(self):
        t = util.gettime()
        print "MessageHandler statistics:"
        print "Number of updates received:", self._updates
        d = t - self._connect_time
        if d > 0:
            updates_per_sec = self._updates / d
        else:
            updates_per_sec = None
        print "Received updates per second:", updates_per_sec


class Player:
    """
    This class creates and controls the player's main window.
    """

    def __init__(self, addr,
                 help_url='http://www.handysoftware.com/cpif/starsim/', 
                 window=None):
        self.addr = addr
        self.help_url = help_url
        self.user = None
        self.dw = DrawingWindow(800, 600, background='black',
                                window=window)
        self.width, self.height = self.dw.width, self.dw.height
        self.dw.title('StarSim')
        self.dw.onKeyDownCall(self._onKeyDown)
        self.dw.onResizeCall(self._onResize)
        self.dw.onDeleteWindowCall(self._onDeleteWindow)
        self.connection = None
        self.msg_id = None
        self.display = None
        self.b1 = None
        self.text1 = None
        self.text2 = None
        self.text_input = None
        self.__update_q = []
        self.__dropped_updates = 0
        self.__update_complete_time = None
        self._stop_flag = False
        self.__rpc_handler = None
        self.__net_connector = None
        self.__net_timer = None
        # Trigger the first redraw
        self.dw.afterDelayCall(TIME_PERIOD, self._update, None)

    def run(self):
        """
        Connects in the background, then runs the DrawingWindow message
        loop.
        """
        self.run = self._alreadyRan
        self.connect()
        self.dw.run()

    def _alreadyRan(self):
        raise Exception("Cannot call run() more than once")

    def connect(self):
        """
        Connect to the server and display the start button.
        """
        self.showMessage("Connecting to " + str(self.addr))
        if isinstance(self.addr, InProcNode):
            self._connect()
        else:
            # Look up IP address of host name in the background
            def _callback(addr):
                print "Completed address lookup:", self.addr, "->", addr
                if isinstance(addr, Exception):
                    error = addr
                    self._handleConnectError(None, error)
                else:
                    self.addr = addr
                    self.dw.afterDelayCall(0, self._connect)
            lookupInetHostInBackground(self.addr, _callback)
        timer = self.__net_timer
        if not timer:
            NET_TIMER_PERIOD = 0.020
            print "Network timer period: %.3f seconds" % NET_TIMER_PERIOD
            timer = self.dw.onTimerCall(NET_TIMER_PERIOD, self._onNetworkTimer)
        self.__net_timer = timer

    def _connect(self):
        factory = self._handleConnect
        if isinstance(self.addr, InProcNode):
            connector, connection = getInProcConnection(self.addr, factory)
        else:
            connector = Connector(self.addr, factory)
            connection = None
        connector.onDisconnectCall(self._handleDisconnect)
        connector.onErrorCall(self._handleConnectError)
        self.__net_connector = connector
        self.connection = connection

    def _onNetworkTimer(self):
        connector = self.__net_connector
        if connector:
            if not connector.poll(0):
                print "Connector stopped"; sys.stdout.flush()
                self.__net_connector = None

    def _handleConnect(self, connection):
        """Called by the rpc framework when the connection to the server
        is completed. Return an instance of the rpc message handler."""
        self.connection = connection
        self.connection.node.onErrorCall(self._handleNetworkError)
        handler = PlayerMessageHandler(connection, self)
        self.__rpc_handler = handler
        self.showMessage("Connected to " + str(self.addr))
        self.dw.afterDelayCall(0, self._initWindow, None)
        return handler

    def _handleConnectError(self, connection, error):
        # called when there is an error while connecting to server
        self.showMessage("Failed to connect to " + repr(self.addr) + ":" +
                         str(error))
        if connection:
            connection.close()
            connection.node.stop()

    def _handleDisconnect(self, unused):
        # called when the other side closes the connection normally
        # unused: the Connection object
        self._disconnect()

    def _handleNetworkError(self, unused, error):
        # called when there is a communication error
        # unused: the Connection object
        msg = "Network error from " + repr(self.addr) + ": " + str(error)
        self._traceFault(error)
        self._disconnect(msg)

    def helloReply(self, result):
        # Called in response to an rpc request
        try:
            status, msg = result.getResult()
        except RemoteException, e:
            status, msg = False, str(e)[:79]
            self._traceFault(e)
        except Disconnected:
            status, msg = False, "Could not connect to server."
        self.showMessage(msg)
        if not status:
            self.dw.afterDelayCall(0, self._initWindow, None)
            return
        proxy = self.connection.getProxy(callback=self.requestShipReply)
        proxy.requestShip()

    def requestShipReply(self, result):
        # Called in response to an rpc request
        try:
            ship_id, msg = result.getResult()
        except (Disconnected, RemoteException):
            self._traceFault()
            self._disconnect()
            return
        self.ship_id = ship_id
        self.createScanDisplay()
        self.showMessage(msg)
        self.dw.onMouseDownCall(self._mouseDown, None)
        self._sendReady()

    def createScanDisplay(self):
        th = self.dw.getTextSize("H")[1]
        y1 = th * 2 + 1
        y2 = self.height - th - 1
        w = y2 - y1 + 1
        x1 = (self.width//2) - (w//2)
        x2 = (self.width//2) + (w//2)
        view.PIXELS_PER_SECTOR = y2 -y1 + 1 # hack
        self.dw.box(x1-1, y1-1, x2-1, y2-1, 'yellow')
        self.display = display.ScanDisplay(x1, y1, x2, y2, self.ship_id)

    def showMessage(self, msg):
        if debug.print_player_messages:
            print msg
        if self.dw:
            self.dw.afterDelayCall(0, self._showMessage, msg)

    def getUpdates(self):
        result = self.__update_q
        self.__update_q = []
        return result

    def addUpdate(self, data):
        self.__update_q.append(data)

    def update(self, data):
        # Called by the server via rpc
        self.addUpdate(data)

    def showMessages(self, msgs):
        """Display messages from the server"""
        # Called by the server via rpc
        for msg in msgs:
            self.showMessage(msg)

    def commandReply(self, result):
        """Generic command reply handler."""
        # Called in response to various rpc requests
        try:
            result.getResult()
        except RemoteException:
            self._traceFault()
        except Disconnected:
            self._disconnect()

    def queryReply(self, result):
        # Called in response to an rpc request
        try:
            init_list = result.getResult()
            self.display.newObjects(init_list)
        except RemoteException:
            self._traceFault()
        except Disconnected:
            self._disconnect()

    def stop(self):
        """Shut down the game player program."""
        # Callable by the server via rpc
        if self.__rpc_handler:
            self.__rpc_handler._printStats()
        print "Dropped updates:", self.__dropped_updates
        print "Goodbye!"; sys.stdout.flush()
        self._disconnect("Disconnecting from server")
        self._stop_flag = True

    def _traceFault(self, e=None):
        if not e:
            e = sys.exc_info()[1]
        sys.stderr.write(str(e))
        sys.stderr.write('\n')

    def _showMessage(self, msg):
        if self.dw:
            if self.msg_id:
                self.dw.delete(self.msg_id)
                self.msg_id = None
            if msg:
                self.msg_id = self.dw.drawText(0, 0, msg, color='white')

    def _update(self, unused=None):
        updates = self.getUpdates()
        if updates and not self._stop_flag:
            data = updates[-1]
            self.__dropped_updates += len(updates) - 1
            self._redraw(data)
        cur_time = util.gettime()
        if self.__update_complete_time is not None:
            delay = TIME_PERIOD - (cur_time - self.__update_complete_time)
        else:
            delay = TIME_PERIOD
        if delay < 0:
            delay = 0
        self.__update_complete_time = cur_time
        if not self._stop_flag:
            self.dw.afterDelayCall(delay, self._update, None)
        else:
            self.dw.close()
            self.dw = None

    def _redraw(self, data):
        ticks, display_list = data
        if not debug.pause_display:
            unknown_ids = self.display.update(self.dw, ticks, display_list)
        else:
            unknown_ids = []
        if unknown_ids:
            try:
                connection = self.connection
                if connection:
                    connection.callMethod('queryInitInfo', (unknown_ids,),
                                               self.queryReply)
            except Disconnected:
                self._traceFault()
                self._disconnect()
        if not debug.pause_player:
            self._sendReady()
        else:
            print "-- skipping sending ready because player is paused --"
            sys.stdout.flush()

    def _sendReady(self):
        connection = self.connection
        if connection:
            try:
                connection.callMethod('ready', (), None)
            except Disconnected:
                self._disconnect()

    def _onKeyDown(self, event, unused=None):
        if self.text_input:
            if self.text_input._onKeyDown(event):
                return
        command = None
        if event.keysym == 'Escape':
            self.stop()
        elif event.keysym == 'Tab':
            if not self.text_input:
                self._inputStarGram()
        elif event.keysym == 'Home':
            command = 'home'
        elif event.keysym == 'Left':
            command = 'left'
        elif event.keysym == 'Right':
            command = 'right'
        elif event.keysym == 'Up':
            command = 'up'
        elif event.keysym == 'Down':
            command = 'down'
        elif event.keysym == 'F1':
            print "Opening help page at", self.help_url
            self._showMessage(
                'Opening StarSim instruction page in web browser...')
            webbrowser.open(self.help_url)
        elif event.keysym == 'F2':
            self._showMessage("F2 doesn't do anything")
        elif event.keysym == 'F3':
            self._showMessage("Load/Unload/Inventory cargo - not working yet")
        elif event.keysym == 'F4':
            self._showMessage("F4 doesn't do anything")
        elif event.keysym == 'F5':
            command = 'fire'
        elif event.keysym == 'F6':
            self._showMessage("F6 doesn't do anything")
        elif event.keysym == 'F7':
            self._showMessage("F7 doesn't do anything")
        elif event.keysym == 'F8':
            self._showMessage("Attach/detach tow line - not working yet")
        elif event.keysym == 'F9':
            self._showMessage("Toggle display mode - not working yet")
        elif event.keysym == 'F10':
            self._showMessage("Measure distance/fuel - not working yet")
        elif event.keysym == 'F11':
            self._showMessage("F11 doesn't do anything")
        elif event.keysym == 'F12':
            self._showMessage("F12 doesn't do anything")
        elif event.keysym == 'PageUp':
            self._showMessage("PageUp not working yet")
        elif event.keysym == 'PageDn':
            self._showMessage("PageDn not working yet")
        elif event.char == 'g':
            debug.garbage_collect_and_report()
        if command and self.connection:
            try:
                self.connection.callMethod(command, (), self.commandReply)
            except Disconnected:
                self._disconnect()

    def _onDeleteWindow(self):
        self.stop()

    def _onResize(self):
        print "Resized to", self.dw.getSize()

    def _disconnect(self, msg="Disconnected from server"):
        #proxy = self.connection and self.connection.getProxy(None)
        #if proxy:
        #    proxy.exit()
        # the above if statement gave me an error, possible Python bug
        connection = self.connection
        self.connection = None
        if connection:
            try:
                proxy = connection.getProxy(callback=None)
                proxy.exit()
            except Disconnected:
                pass
            connection.close()
            connection.node.stop()
        self.dw.afterDelayCall(0, self._clearStartButton, None)
        self.showMessage(msg)

    def _initWindow(self, dummy=None):
        x, y = self.width/2, self.height/4
        self.text1 = self._centerText(x, y, "Welcome to StarSim")
        x, y = self.width/2, self.height * 0.75
        self.text2 = self._centerText(x, y, "Click the button to begin")
        x, y = self.width/2, self.height/2
        self.b1 = self.dw.circle(x, y, 50, 'green', fill='dark green',
                                 width=5)
        self.dw.onClickCall(self.b1, self._start, None)

    def _initWindowEvent(self, unused):
        self._initWindow()

    def _clearStartButton(self, dummy=None):
        self.dw.delete(self.text1); self.text1 = None
        self.dw.delete(self.text2); self.text2 = None
        self.dw.delete(self.b1); self.b1 = None

    def _clearStartButtonEvent(self, unused):
        self._clearStartButton()

    def _start(self, dummy, unused=None):
        # dummy: event
        self._showMessage('')
        self._clearStartButton()
        s = "Enter your user name: "
        w, h = self.dw.getSize()
        tw, th = self.dw.getTextSize(s)
        x = w/2 - tw
        y = h/2 - th/2
        self.text_input = TextInput(self.dw, x, y, s, self._userInputDone)

    def _userInputDone(self, user):
        self.user = user
        self.text_input.close()
        self.text_input = None
        if not self.user:
            self._initWindow()
            return
        self.showMessage("Logging in as " + repr(self.user))
        try:
            proxy = self.connection.getProxy(self.helloReply)
            proxy.hello(self.user)
        except (Disconnected, RemoteException):
            self._traceFault()
            self._disconnect()

    def _inputStarGram(self):
        s = "Enter StarGram message: "
        tw, th = self.dw.getTextSize(s)
        x, y = 0, th
        self.text_input = TextInput(self.dw, x, y, s, self._starGramDone)

    def _starGramDone(self, text):
        self.text_input.close()
        self.text_input = None
        try:
            proxy = self.connection.getProxy(None)
            proxy.sendStarGram(text)
        except (Disconnected, RemoteException):
            self._traceFault()
            self._disconnect()

    def _centerText(self, x, y, s):
        w, h = self.dw.getTextSize(s)
        return self.dw.drawText(x - w/2, y - h/2, s, color='white')

    def _mouseDown(self, event, unused):
        obj_id, loc = self.display.pickObject(event.x, event.y)
        if loc:
            x, y = loc
            print "Clicked on: %s at (%+6.2f, %+6.2f)" % (obj_id, x, y)
        if event.button_num == 3: # right button
            command = 'setSelection'
        else: # middle or left button
            command = 'setDestination'
        if self.ship_id and self.connection:
            if obj_id is not None:
                param = obj_id
            else:
                param = loc
            try:
                self.connection.callMethod(command, (param,), 
                                           self.commandReply)
            except Disconnected:
                self._disconnect()


class TextInput:

    def __init__(self, dw, x, y, prompt='', callback=None):
        self.dw = dw
        self.x = x
        self.y = y
        self.prompt = prompt
        self.callback = callback
        self.text = ''
        self.text_id = None
        self.cursor_id = None
        self._refresh()

    def close(self):
        if self.text_id:
            self.dw.delete(self.text_id)
        if self.cursor_id:
            self.dw.delete(self.cursor_id)
        self.text_id = None
        self.cursor_id = None

    def _refresh(self):
        if self.text_id:
            self.dw.delete(self.text_id)
        if self.cursor_id:
            self.dw.delete(self.cursor_id)
        s = "%s%s" % (self.prompt, self.text)
        tw, th = self.dw.getTextSize(s)
        aw, ah = self.dw.getTextSize('A')
        self.text_id = self.dw.drawText(self.x, self.y, s, color='white')
        self.cursor_id = self.dw.box(self.x + tw, self.y,
                                     self.x + tw + aw, self.y + ah, 'white')

    def _onKeyDown(self, event, data=None):
        """Return True iff this control "eats" the event."""
        processed = False
        char = event.char
        if char and ord(char) >= 32 and ord(char) <> 0x7f:
            self.text += char
            processed = True
        elif event.keysym in ('BackSpace', 'Delete'):
            self.text = self.text[:-1]
            processed = True
        elif event.keysym == 'Return':
            if self.callback:
                self.callback(self.text)
                return True
        elif event.keysym == 'Escape':
            if self.callback:
                self.callback(None)
                return True
        if processed:
            self._refresh()
        return processed


# end-of-file
