# starsim/player/display.py
"""
Display module -- classes to handle each of StarSim's display modes.
"""

import math
import sys

from cpif.graphics import Transform2D

from starsim.player import view


class ScanDisplay:

    def __init__(self, x1, y1, x2, y2, ship_id):
        """
        x1, y1   -- Upper left pixel coordinates of display
        x2, y2   -- Lower left pixel coordinates + 1
        ship_id  -- ID of ship that is sending us scan info
        """
        self.width = x2 - x1
        self.height = y2 - y1
        self.cx = (x1 + x2) // 2
        self.cy = (y1 + y2) // 2
        self.ship_id = ship_id
        self.objects = {}
        self.time_id = None
        self.__objects_to_delete = []

    def _deleteObjectLater(self, obj):
        self.__objects_to_delete.append(obj)

    def _getObjectsToDelete(self):
        result = self.__objects_to_delete
        self.__objects_to_delete = []
        return result

    def update(self, dw, ticks, display_list):
        """
        Update the display with the latest display items.
        Return a list of the IDs from display_list that were unknown.
        """
        unknown_ids = []
        display_objs = []
        objects = self.objects.copy()
        for items in display_list:
            id = items[0]
            stuff = items[1:]
            display_obj = objects.pop(id, None)
            if not display_obj:
                unknown_ids.append(id)
            else:
                display_obj.update(*stuff)
                display_objs.append(display_obj)
        # delete objects that are no longer being displayed
        for id in objects.iterkeys():
            display_obj = self.objects.pop(id)
            display_obj.delete(dw)
        del objects
        # delete other leftover objects
        for obj in self._getObjectsToDelete():
            obj.delete(dw)
        if self.ship_id:
            ship = self.objects.get(self.ship_id)
            if ship:
                loc = ship.loc
                self._drawObjects(dw, loc, display_objs)
                status = "%9d (%+6.2f, %+6.2f)" % (ticks, loc[0], loc[1])
            else:
                status = "%9d (?, ?)" % ticks
        else:
            status = "%9d" % (ticks,)
        h = dw.getTextSize(status)[1]
        x = self.cx - (self.width//2)
        y = self.cy + (self.height//2) - h
        if self.time_id:
            dw.delete(self.time_id)
        self.time_id = dw.drawText(x, y, status, color='white')
        return unknown_ids

    def _drawObjects(self, dw, loc, display_objs):
        pov = self._getPointOfView(loc)
        for display_obj in display_objs:
            display_obj.draw(dw, pov)

    def _getPointOfView(self, loc):
        pov = Transform2D()
        pov.translate(-loc[0], -loc[1])
        pov.scale(view.PIXELS_PER_SECTOR, -view.PIXELS_PER_SECTOR)
        pov.translate(self.cx, self.cy)
        return pov

    def newObjects(self, init_list):
        # Called by RPC
        for items in init_list:
            if not items:
                continue
            class_name, id = items[:2]
            class_ = getattr(view, class_name, None)
            if class_ and not issubclass(class_, view.ViewBase):
                sys.stderr.write(
                    "!! Security violation: attempt to call %s\n" %
                    repr(class_))
                class_ = None
            if not class_:
                display_obj = view.Unknown(id, class_name)
            else:
                try:
                    display_obj = class_(id, *items[2:])
                except TypeError:
                    print >> sys.stderr, "** Error creating View:",
                    print >> sys.stderr, class_, id, items[2:]
                    display_obj = view.Unknown(id, class_name)
            old_obj = self.objects.pop(id, None)
            if old_obj:
                self._deleteObjectLater(old_obj)
            self.objects[id] = display_obj

    def pickObject(self, x, y):
        """
        x, y: Pixel location in window coordinates

        Return (pick_id, (ux, uy))
        pick_id:
            The ID of the object at pixel location (x, y), or None if there
            is no such object. If (x, y) is on or near multiple objects,
            return the ID of the object whose center is nearest to (x, y).
        (ux, uy):
            The universe coordinates of the mouse click location
        """
        matches = []
        for display_obj in self.objects.itervalues():
            bbox = getattr(display_obj, 'bbox', None)
            if not bbox:
                continue
            (minx, miny), (maxx, maxy) = display_obj.bbox
            if (minx <= x and x <= maxx and miny <= y and y <= maxy):
                cx = (minx + maxx) / 2.
                cy = (miny + maxy) /2.
                matches.append((display_obj.id, cx, cy))
        result_id = None
        min_d = None
        for id, cx, cy in matches:
            d = math.sqrt(cx*cx + cy*cy)
            if min_d is None or d < min_d:
                min_d = d
                result_id = id
        if result_id:
            result_obj = self.objects.get(result_id)
            loc = result_obj.loc
        else:
            ship = self.objects.get(self.ship_id)
            if ship:
                # Transform pixel coordinates to universe coordinate
                t = Transform2D()
                angle = 0.0
                scale = 1.0 / view.PIXELS_PER_SECTOR
                t.set(-angle, scale, -scale, ship.loc[0], ship.loc[1])
                px, py = (x - self.cx, y - self.cy)
                loc = t.transform([(px, py)])[0]
            else:
                # Can't transform to universe coordinates if we don't know
                # where in the universe we are located.
                loc = None
        return result_id, loc


# end-of-file
