# cpif/turtle.py
# Copyright (C) 2003-2005 by David Handy
# See the LICENSE.txt file for terms and conditions of use.
"""
Turtle Graphics module - simplified LOGO-style graphics.

This module works just like the standard Python turtle module, but with
several improvements:

* More reliable when used from IDLE's Python Shell
* More animation
* degrees() and radians() work consistently
* Docstrings; use help() on this module
* Built-in unit tests; run this module's test() function

This module uses cpif.graphics for drawing, which in turn uses Tkinter.
"""

# I hate the "import *", but that is what the standard Python turtle module
# does, and I want to be compatible.
from math import *
import sys
import threading

from cpif.graphics import DrawingWindow

_dw = None # global drawing window
_pen = None # global turtle Pen


class Error(Exception):
    pass


SECONDS_PER_STEP = 0.02
PIXELS_PER_STEP = 2.0


def _killTurtle(*params, **kwparams):
    class TurtleDead(Exception):
        pass
    raise TurtleDead("Window killed while turtle was drawing")


def _killInstance(obj):
    for name in dir(obj):
        attr = getattr(obj, name)
        if callable(attr) and not name.startswith('_'):
            setattr(obj, name, _killTurtle)


class _AnimationTimer(object):

    def __init__(self, steps):
        assert steps >= 0
        self.delay = SECONDS_PER_STEP
        self.__steps = steps
        self.__event = threading.Event()
        self.__time_limit = max(steps * self.delay * 10.0, 30.0)
        self.__is_killed = False

    def done(self):
        """Return True iff there are no more steps."""
        return self.__steps <= 0

    def kill(self):
        self.__is_killed = True
        self.__steps = 0
        self.__event.set()

    def step(self):
        """Decrement the step count, set event if done."""
        self.__steps -= 1
        if self.__steps <= 0:
            self.__event.set()

    def wait(self):
        self.__event.wait(self.__time_limit)
        if self.__is_killed:
            _killTurtle()


class _AnimationBase(object):

    def __init__(self, pen, steps):
        self._pen = pen
        self._timer = _AnimationTimer(steps)

    def _getDelay(self):
        return self._timer.delay

    delay = property(_getDelay)

    def next(self):
        """
        Move the turtle to the next step in the animation sequence, 
        tracing a line if drawing is turned on.

        Return the next animation object (self or None).
        """
        self.step()
        if self.done():
            return None
        return self

    def done(self):
        return self._timer.done()

    def step(self):
        return self._timer.step()

    def kill(self):
        self._timer.kill()

    def wait(self):
        return self._timer.wait()


class _LineAnimation(_AnimationBase):
    """Move in a straight line from start position to end position,
    possibly stretching a line behind the turtle."""

    def __init__(self, pen, start, end):
        self.start = start
        self.end = end
        ###
        self._pos = start
        x0, y0 = start
        x1, y1 = end
        xx = x1 - x0
        yy = y1 - y0
        steps = int(sqrt((xx * xx) + (yy * yy)) / PIXELS_PER_STEP)
        super(_LineAnimation, self).__init__(pen, steps)
        if steps:
            self.dx = xx / steps
            self.dy = yy / steps
        else:
            self.dx = xx
            self.dy = yy
        if pen._drawing:
            self.item = pen._dw.line(x0, x0, x0, x0, pen._color,
                                     width=pen._width)
        else:
            self.item = None

    def next(self):
        result = super(_LineAnimation, self).next()
        if not self.done():
            x, y = self._pos
            self._pos = (x + self.dx, y + self.dy)
        else:
            self._pos = self.end
        pen = self._pen
        if self.item:
            pen._dw.setCoords(self.item, [self.start, self._pos])
        pen._draw_turtle(self._pos)
        return result


class _CircleAnimation(_AnimationBase):
    """Move the turtle in a circular motion, possibly extending an arc."""

    def __init__(self, pen, x, y, radius, start, extent):
        """
        pen: turtle Pen
        x, y: arc/circle center
        radius: arc/circle radius, >= 0
        start: starting angle in degrees
        extent: degrees to draw, +/- clockwise
        """
        fullcircle = pen._fullcircle
        circumference = abs(2 * pi * radius)
        pixels = abs(circumference * (extent / fullcircle))
        steps = int(pixels / PIXELS_PER_STEP)
        super(_CircleAnimation, self).__init__(pen, steps)
        self.x = x
        self.y = y
        self.radius = radius
        self.start = start
        self.extent = extent
        self._a = 0 # current extent angle
        self._da = fullcircle * (PIXELS_PER_STEP / circumference)
        if extent < 0:
            self._da = -self._da
        if pen._drawing:
            self.item = pen._dw.arc(x, y, radius, pen.to_degrees(start), 0, 
                                    pen._color,
                                    width=pen._width,
                                    style=DrawingWindow.ARC)
        else:
            self.item = None

    def next(self):
        result = super(_CircleAnimation, self).next()
        if not self.done():
            self._a += self._da
        else:
            self._a = self.extent
        pen = self._pen
        fullcircle = pen._fullcircle
        x = self.x
        y = self.y
        radius = self.radius
        if self.item:
            pen._dw.delete(self.item)
            if abs(self._a) < fullcircle:
                self.item = pen._dw.arc(x, y, radius,
                                        pen.to_degrees(self.start),
                                        pen.to_degrees(self._a), 
                                        pen._color,
                                        width=pen._width,
                                        style=DrawingWindow.ARC)
            else:
                self.item = pen._dw.circle(x, y, radius, pen._color,
                                           width=pen._width)
        angle = self.start + self._a
        angle_rad = pen.to_radians(angle)
        x1 = x + abs(radius) * cos(angle_rad)
        y1 = y - abs(radius) * sin(angle_rad)
        if radius >= 0.0:
            pen._angle = angle + (fullcircle / 4.0)
        else:
            pen._angle = angle - (fullcircle / 4.0)
        pen._position = x1, y1
        pen._draw_turtle()
        return result


class RawPen(object):

    def __init__(self, dw):
        self._dw = dw
        self._tracing = True
        self._arrow = None
        self._animation = None
        self._fullcircle = 360.0
        self._angle = 0
        self.degrees()
        self.reset()

    def _drawNext(self):
        # Draw the next step in an animation sequence
        animation = self._animation
        if not animation:
            return
        animation = animation.next()
        if animation:
            self._dw.afterDelayCall(animation.delay, self._drawNext)
        self._animation = animation

    def to_degrees(self, angle):
        """
        Convert an angle to degrees. angle parameter is in the current mode
        (degrees or radians).
        """
        return angle * self._deg_per_unit

    def to_radians(self, angle):
        """
        Convert an angle to radians. angle parameter is in the current mode
        (degrees or radians).
        """
        return angle * self._rad_per_unit

    def degrees(self, fullcircle=360.0):
        """
        Set the angular units used by this pen. fullcircle is by default
        360, meaning degrees. This can cause the pen to have any angular
        units whatever: give fullcircle 2*pi for radians, or 400 for
        gradians.
        """
        old_fullcircle = self._fullcircle
        self._fullcircle = fullcircle
        self._deg_per_unit = 360.0 / fullcircle
        self._rad_per_unit = (2.0 * pi) / fullcircle
        self._angle = self._angle * (self._fullcircle / old_fullcircle)

    def radians(self):
        self.degrees(fullcircle=2*pi)

    def reset(self):
        width, height = self._dw.getSize()
        if width <= 1:
            width = self._dw.width
        if height <= 1:
            height = self._dw.height
        self._origin = float(width)/2.0, float(height)/2.0
        self._position = self._origin
        self._angle = 0.0
        self._drawing = 1
        self._width = 1
        self._color = "black"
        self._filling = 0
        self._path = []
        self._tofill = []
        self.clear()

    def clear(self):
        self.fill(0)
        self._dw.clear()
        self._draw_turtle()

    def tracer(self, flag):
        self._tracing = flag
        if not self._tracing:
            self._delete_turtle()
        self._draw_turtle()

    def forward(self, distance):
        x0, y0 = start = self._position
        x1 = x0 + distance * cos(self._angle*self._rad_per_unit)
        y1 = y0 - distance * sin(self._angle*self._rad_per_unit)
        self._goto(x1, y1)

    def backward(self, distance):
        self.forward(-distance)

    def left(self, angle):
        self._angle = (self._angle + angle) % self._fullcircle
        self._draw_turtle()

    def right(self, angle):
        self.left(-angle)

    def up(self):
        self._drawing = 0

    def down(self):
        self._drawing = 1

    def width(self, width):
        self._width = float(width)

    def color(self, *args):
        def _checkColor():
            if isinstance(colorparam, basestring):
                return colorparam
            try:
                red, green, blue = map(int, colorparam) # [int(p) for p in colorparam]
                validrange = _VALID_COLOR_RANGE
                if red not in validrange or green not in validrange or \
                   blue not in validrange:
                    raise ValueError()
                return '#%2.2x%2.2x%2.2x' % (red, green, blue)
            except (TypeError, ValueError):
                raise ValueError(_COLOR_USAGE_MSG)
        if not args:
            raise Error("no color arguments")
        if len(args) == 1:
            color = args[0]
            if isinstance(color, basestring):
                # Test the color first
                self._dw.line(0, 0, 0, 0, color)
                # Now set the color for real
                self._set_color(color)
                return
            try:
                r, g, b = color
            except:
                raise Error("bad color sequence: %r" % (color,))
        else:
            try:
                r, g, b = args
            except:
                raise Error("bad color arguments: %r" % (args,))
        assert 0 <= r <= 1
        assert 0 <= g <= 1
        assert 0 <= b <= 1
        x = 255.0
        y = 0.5
        self._set_color("#%02x%02x%02x" % (int(r*x+y), int(g*x+y), int(b*x+y)))

    def _set_color(self,color):
        self._color = color
        self._draw_turtle()

    def write(self, arg, move=0):
        x, y = self._position
        x = x-1 # correction -- calibrated for Windows
        w, h = self._dw.getTextSize(arg)
        item = self._dw.drawText(x, y - h, arg, color=self._color)
        if move:
            self._goto(x + w, y)
        self._draw_turtle()

    def fill(self, flag):
        if self._filling:
            path = tuple(self._path)
            if len(path) > 2:
                item = self._dw.polygon(path, self._color, fill=self._color)
            for method, params, kwparams in self._tofill:
                # Redraw all circles, with fill parameter set
                kwparams['fill'] = self._color
                method(*params, **kwparams)
        self._path = []
        self._tofill = []
        self._filling = flag
        if flag:
            self._path.append(self._position)
        #self.forward(0) ## Why was this line in the original turtle module?

    def circle(self, radius, extent=None):
        if extent is None:
            extent = self._fullcircle
        x0, y0 = self._position
        xc = x0 - radius * sin(self._angle * self._rad_per_unit)
        yc = y0 - radius * cos(self._angle * self._rad_per_unit)
        if radius >= 0.0:
            start = self._angle - (self._fullcircle / 4.0)
        else:
            start = self._angle + (self._fullcircle / 4.0)
            extent = -extent
        kwparams = {'width': self._width}
        if abs(extent) >= self._fullcircle:
            method = self._dw.circle
            params = (xc, yc, radius, self._color)
        else:
            method = self._dw.arc
            params = (xc, yc, radius, start, extent, self._color)
            kwparams['style'] = DrawingWindow.ARC
        angle = start + extent
        x1 = xc + abs(radius) * cos(angle * self._rad_per_unit)
        y1 = yc - abs(radius) * sin(angle * self._rad_per_unit)
        self._angle = (self._angle + extent) % self._fullcircle
        self._position = x1, y1
        if self._tracing and not self._dw.inDrawingThread():
            animation = _CircleAnimation(self, xc, yc, radius, start, extent)
            self._animation = animation
            self._dw.afterDelayCall(animation.delay, self._drawNext)
            animation.wait()
        else:
            if self._drawing:
                method(*params, **kwparams)
        if self._filling:
            if kwparams.get('style'):
                style = DrawingWindow.CHORD
                kwparams['style'] = style
            self._tofill.append((method, params, kwparams))
            self._path.append(self._position)
        self._draw_turtle()

    def heading(self):
        return self._angle

    def setheading(self, angle):
        self._angle = angle
        self._draw_turtle()

    def window_width(self):
        return self._dw.width

    def window_height(self):
        return self._dw.height

    def position(self):
        x0, y0 = self._origin
        x1, y1 = self._position
        return [x1-x0, -y1+y0]

    def setx(self, xpos):
        x0, y0 = self._origin
        x1, y1 = self._position
        self._goto(x0+xpos, y1)

    def sety(self, ypos):
        x0, y0 = self._origin
        x1, y1 = self._position
        self._goto(x1, y0-ypos)

    def goto(self, *args):
        if len(args) == 1:
            try:
                x, y = args[0]
            except:
                raise Error, "bad point argument: %r" % (args[0],)
        else:
            try:
                x, y = args
            except:
                raise Error, "bad coordinates: %r" % (args[0],)
        x0, y0 = self._origin
        self._goto(x0+x, y0-y)

    def _destroy(self):
        # Called right before the window is destroyed.
        # Stop any animation in progress
        animation = self._animation
        if animation:
            animation.kill()
        # Disable all methods on this object
        _killInstance(self._dw)
        _killInstance(self)

    def _goto(self, x1, y1):
        x0, y0 = self._position
        self._position = tuple(map(float, (x1, y1)))
        if self._tracing and not self._dw.inDrawingThread():
            animation = _LineAnimation(self, (x0, y0), (x1, y1))
            self._animation = animation
            self._dw.afterDelayCall(animation.delay, self._drawNext)
            animation.wait()
        else:
            if self._drawing:
                self._dw.line(x0, y0, x1, y1, self._color, width=self._width)
        if self._filling:
            self._path.append(self._position)
        self._draw_turtle()

    def _draw_turtle(self, position=()):
        if not self._tracing:
            return
        if not position:
            position = self._position
        x, y = position
        distance = 8
        dx = distance * cos(self._angle*self._rad_per_unit)
        dy = -distance * sin(self._angle*self._rad_per_unit)
        self._delete_turtle()
        points = [(x, y), 
                  (x - dx - dy, y - dy + dx), 
                  (x - dx + dy, y - dy - dx)]
        self._arrow = self._dw.polygon(points, self._color)

    def _delete_turtle(self):
        if self._arrow:
            self._dw.delete(self._arrow)
        self._arrow = None


class Pen(RawPen):

    def __init__(self, width=355, height=249):
        global _dw
        if _dw is None:
            self.dw = DrawingWindow(width, height)
            self.dw.title("Turtle Window")
            self.dw.onDeleteWindowCall(self._destroy)
            _dw = self.dw
        else:
            self.dw = _dw
        super(Pen, self).__init__(self.dw)

    def _destroy(self):
        super(Pen, self)._destroy()
        global _dw, _pen
        if self.dw is _dw:
            _pen = None
            _dw = None


def _getpen():
    global _pen
    pen = _pen
    if not pen:
        _pen = pen = Pen()
    return pen


# The standard turtle API

def degrees():
    """
    Set angle measurement units to degrees.
    """
    _getpen().degrees()

def radians():
    """
    Set angle measurement units to radians.
    """
    _getpen().radians()

def reset():
    """
    Clear the window, reset the turtle color and other variables, and move
    the turtle back to the center, facing right.
    """
    _getpen().reset()

def clear():
    """
    Clear the window, leaving the turtle in the same position.
    """
    _getpen().clear()

def tracer(flag):
    """
    Turn tracing on or off, depending on whether flag is True or False. When
    tracing is on, the turtle draws slowly so you can see it better.
    """
    _getpen().tracer(flag)

def forward(distance):
    """
    Move turtle forward 'distance' pixels. Draw line if pen is down. (Pen
    starts out down.)
    """
    _getpen().forward(distance)

def backward(distance):
    """
    Move turtle backward 'distance' pixels. Draw line if pen is down. (Pen
    starts out down.)
    """
    _getpen().backward(distance)

def left(angle):
    """
    Turn the turtle left 'angle' degrees. (180 degrees turn means face the
    exact opposite direction.)
    """
    _getpen().left(angle)

def right(angle):
    """
    Turn the turtle right 'angle' degrees (180 degrees turn means face the
    exact opposite direction.)
    """
    _getpen().right(angle)

def up():
    """
    Raise the pen up, stop drawing. (The turtle can still move.)
    """
    _getpen().up()

def down():
    """
    Put the pen down, start drawing. (Pen starts in down position.)
    """
    _getpen().down()

def width(width):
    """
    Draw lines 'width' pixels wide. Default is 1.
    """
    _getpen().width(width)

def color(*args):
    """
    color(s)
    color((r, g, b))
    color(r, g, b)
    -> set new drawing color

    s is a color name, such as 'red', 'green', or 'blue' (include the quote
    marks). Or use an RGB color tuple with red, green, and blue levels
    between 0.0 and 0.1. (0, 0, 0) is black and (1, 1, 1) is white.
    """
    _getpen().color(*args)

def write(text, move=0):
    """
    write(text) -> write text at current turtle location
    write(text, move=True) -> write text and move turtle to end of text
    """
    _getpen().write(text, move)

def fill(flag):
    """
    Run fill(True) before drawing a path, and fill(False) when done drawing,
    and the path will be filled on the inside with the  current color.
    """
    _getpen().fill(flag)

def circle(radius, extent=None):
    """
    circle(radius)
    circle(radius, extent)
    -> Draw a circle to the left of the turtle.

    If extent is given, draws that many degrees (out of 360) of the circle.
    If radius is negative, the circle is drawn to the right of the turtle.
    """
    _getpen().circle(radius, extent)

def goto(*args):
    """
    Move the turtle to the given x and y pixel coordinates, where (0, 0) is
    the center of the window. If the pen is down (the default) a line is
    drawn. The turtle remains pointing at the same angle as before the move.
    """
    _getpen().goto(*args)

def heading():
    """
    Return the current direction of the turtle.
    """
    return _getpen().heading()

def setheading(angle):
    """
    Set the heading, or direction, of the turtle in degrees
    counter-clockwise from the right. (0 is right, 90 is up, 180 is left,
    270 is down. Try negative numbers also.)
    """
    _getpen().setheading(angle)

def position():
    """
    Return the current (x, y) location of the turtle.
    """
    return _getpen().position()

def window_width():
    return _getpen().window_width()

def window_height():
    return _getpen().window_height()

def setx(xpos):
    _getpen().setx(xpos)

def sety(ypos):
    _getpen().sety(ypos)


# Extensions to the turtle API

def to_degrees(angle):
    """Convert an angle to degrees. Assume angle is in the current mode
    (degrees or radians).
    """
    return _getpen().to_degrees(angle)

def to_radians(angle):
    """Convert an angle to radians. Assume angle is in the current mode
    (degrees or radians).
    """
    return _getpen().to_radians(angle)


def demo():
    """
    Clears the window and draws a test pattern.
    """
    reset()
    tracer(1)
    up()
    backward(100)
    down()
    # draw 3 squares; the last filled
    width(3)
    for i in range(3):
        if i == 2:
            fill(1)
        for j in range(4):
            forward(20)
            left(90)
        if i == 2:
            color("maroon")
            fill(0)
        up()
        forward(30)
        down()
    width(1)
    color("black")
    # move out of the way
    tracer(0)
    up()
    right(90)
    forward(100)
    right(90)
    forward(100)
    right(180)
    down()
    # some text
    write("startstart", 1)
    write("start", 1)
    color("red")
    # staircase
    for i in range(5):
        forward(20)
        left(90)
        forward(20)
        right(90)
    # filled staircase
    fill(1)
    for i in range(5):
        forward(20)
        left(90)
        forward(20)
        right(90)
    fill(0)
    # more text
    write("end")
    if __name__ == '__main__':
        _dw.run()


__test__ = {
    "call_main_functions":
        """
        >>> degrees()
        >>> to_degrees(360)
        360.0
        >>> print "%.2f" % to_radians(360)
        6.28
        >>> radians()
        >>> print "%.0f" % to_degrees(6.28)
        360
        >>> print "%.2f" % to_radians(6.28)
        6.28
        >>> degrees()
        >>> reset()
        >>> clear()
        >>> tracer(True)
        >>> tracer(False)
        >>> tracer(True)
        >>> forward(100)
        >>> backward(100)
        >>> left(90)
        >>> right(90)
        >>> up()
        >>> down()
        >>> width(3)
        >>> width(1)
        >>> color('red')
        >>> color((0.5, 0.75, 0.25))
        >>> write("hello")
        >>> fill(True)
        >>> circle(25)
        >>> fill(False)
        >>> circle(50, 180)
        >>> goto(10, 20)
        >>> goto((0, 0))
        """,
    "test_circle":
        """
        >>> reset()
        >>> circle(50, 180)
        >>> print "%.0f" % heading()
        180
        >>> x, y = position()
        >>> print "%d, %d" % (x, y)
        0, 100
        >>> radians()
        >>> circle(50, pi)
        >>> print "%.2f" % (heading() % (2 * pi))
        0.00
        >>> x, y = position()
        >>> print "%d, %d" % (x, y)
        0, 0
        >>> degrees()
        """,
    "test_fill":
        """
        >>> reset()
        >>> color('blue')
        >>> fill(True)
        >>> forward(80)
        >>> right(120)
        >>> forward(80)
        >>> right(120)
        >>> forward(80)
        >>> fill(False)
        >>> import time; time.sleep(0.5)
        >>> x, y = position()
        >>> print "%d, %d" % (x, y)
        0, 0
        >>> print "%.0f" % heading()
        120
        """,
    "test_rotate":
        """
        >>> reset()
        >>> heading()
        0.0
        >>> left(90)
        >>> heading()
        90.0
        """,
    "test_square":
        """
        >>> reset()
        >>> forward(100)
        >>> left(90)
        >>> forward(100)
        >>> x, y = position()
        >>> print "%d, %d" % (x, y)
        100, 100
        >>> heading()
        90.0
        >>> radians()
        >>> left(pi / 2.0)
        >>> forward(100)
        >>> left(pi / 2.0)
        >>> forward(100)
        >>> x, y = position()
        >>> print "%d, %d" % (x, y)
        0, 0
        >>> print "%.2f" % heading()
        4.71
        >>> degrees()
        >>> print "%.0f" % heading()
        270
        """,
    }


def test():
    # Try to put turtle window in known starting state
    reset()
    degrees()
    color('black')
    # Run the module's doctests (see the __test__ dictionary above)
    import doctest
    import sys
    doctest.testmod(sys.modules[__name__])


if __name__ == '__main__':
    test()
    if _pen:
        _pen._dw.run()

# end-of-file
