# music.py
# Copyright (C) 2004 by David Handy
"""
Module for converting musical notes to sound frequencies and durations.
"""

from math import pow, log, floor, ceil

from cpif.sound import beep, gentones, mix, mul


def noteToFrequency(note_name):
    """
    Convert a note name to a sound frequency.

    note_name:
        A string identifying a note. The first character of the string is a
        letter from A to G, upper or lower case, optionally followed by a #
        for sharp or a - for float, followed by an octave number 0 through
        9. None or an empty string results in zero frequency.

        'C4' is middle C, and the first note in octave 4.
        'C#4' is middle C sharp.
        'B-3' is B flat just below middle C.

    Background:
        from http://www.boulder.nist.gov/timefreq/general/glossary.htm :

        A440 (sometimes called A4) is the 440 Hz tone that serves as the
        internationally recognized standard for musical pitch. A440 is the
        musical note A above middle C. Since 1939, it has served as the
        audio frequency reference for the calibration of pianos, violins,
        and other musical instruments.

    Counting sharps and flats, there are 12 notes in an octave. Going from
    'C4' to 'C5' (middle C to the next highest C) doubles the frequency. The
    frequencies of the notes in between go up by factor of 2^1/12 each step.
    """
    return 440.0 * pow(2.0, (noteNum(note_name) - 69.0)/12.0)


def freqToNote(frequency):
    """
    Return the MIDI note number corresponding to a frequency.
    Note number 69 is middle C.

    The number returned is a fractional floating-point number. In MIDI
    files, however, the note numbers are always integers.
    """
    return 12.0 * (log(frequency/440.0) / log(2.0)) + 69.0


notes1 = ['C','C#','D','D#','E','F','F#','G','G#','A','A#','B']
notes2 = ['C','D-','D','E-','E','F','G-','G','A-','A','B-','B']

# lowest note : C0 = MIDI note  12
# highest note: G9 = MIDI note 127
# piano goes from A0=21 to C8=108


def noteName(note_num):
    """
    Convert from a MIDI note number to a note name.

    note_num: A MIDI note number, from 12 to 127 inclusive. 69 is middle C.

    Return the note name. See noteToFrequency() for a description of note
    names.
    """
    if note_num < 12 or note_num > 127:
        raise ValueError("MIDI note number out of range: %s" % note_num)
    octave = int((note_num - 12) / 12.0)
    step = int(note_num - 12) % 12
    note = notes1[step]
    return "%s%d" % (note, octave)


def noteNum(note_name):
    """
    Return the MIDI note number corresponding to a note name.
    
    note_name:
        A string containing the note name. See noteToFrequency() for
        a description of note names.

    Return the MIDI note number. 69 is middle C.
    """
    if len(note_name) == 2:
        note = note_name[0].upper()
        octaveChar = note_name[1]
    elif len(note_name) == 3:
        note = note_name[0:2].upper()
        octaveChar = note_name[2]
    else:
        assert None, ('Invalid note name length  note_name=%s' %
                      repr(note_name))
        return None
    try:
        octave = int(octaveChar)
    except ValueError:
        assert None, 'Invalid octave number: %s' % octaveChar
        return None
    try:
        step = notes1.index(note)
    except ValueError:
        try:
            step = notes2.index(note)
        except ValueError:
            assert None, 'Invalid note name: %s' % note
            return None
    return 12 + (octave*12) + step


def frequencyToNoteName(frequency):
    """This works similar to the Frequency to Musical Note Converter
    at http://www.phys.unsw.edu.au/music/note/ , except that the input range
    is 16.35 Hz (C0) through 12543.85 Hz (G9)."""
    assert (frequency >= 16.35 and frequency <= 12543.854), \
        "Frequency out of range: %s" % frequency
    note_num = freqToNote(frequency)
    if (note_num - floor(note_num)) <= (ceil(note_num) - note_num):
        n = floor(note_num)
    else:
        n = ceil(note_num)
    cents = int((note_num - n) * 100)
    if cents <> 0:
        return "%s%+d%%" % (NoteName(n), cents)
    return NoteName(n)


def setTempo(beats_per_minute):
    """
    Set the overall musical tempo.
    """
    _music_params.setTempo(beats_per_minute)


def setVolume(amplitude):
    """
    Set the overall music volume.

    amplitude: loudness on a scale of 0.0 to 1.0
    """
    _music_params.setAmplitude(amplitude)


def playNote(note_name, note_length):
    """
    Play a musical note by generating a beep of the correct pitch.

    note_name:
        A string identifying a note. The first character of the string is a
        letter from A to G, upper or lower case, optionally followed by a #
        for sharp or a - for float, followed by an octave number 0 through
        9. None or an empty string results in zero frequency.

        'C4' is middle C, and the first note in octave 4.
        'C#4' is middle C sharp.
        'B-3' is B flat just below middle C.
        '' (empty string) is a rest.

    note_length:
        The fractional length of the note compared to the classical "whole"
        note. 0.25 is a quarter note, 0.5 is a half note, etc.

    see setTempo() and setVolume() in this module for ways to influence the
    effect of note_length and the amplitude of the generated tones.
    """
    if not note_name:
        frequency = None
    else:
        frequency = noteToFrequency(note_name)
    time_sec = _music_params.noteLengthToTimeSec(note_length)
    amplitude = _music_params.getAmplitude()
    beep(frequency, time_sec, amplitude)


def soundOfMusic(note_list):
    """
    Generate a sound object (from the cpif.sound module) from a sequence of
    musical notes. Each item in the sequence is a tuple in the form
    
        (note_name, note_length)

    Where note_name and note_length have the same meaning as in the
    playNote() function in this module.
    
    You can use cpif.sound.play() to play the resulting sound.
    You can use save() on the resulting sound to save it to a WAV file.
    You may wish to use the mix(), join(), and mul() functions from the
    cpif.sound module to modify and combine several music sounds into a
    larger composition.
    """
    amplitude = _music_params.getAmplitude()
    sound_list = []
    for note_name, note_length in note_list:
        if not note_name:
            frequency = None
        else:
            frequency = noteToFrequency(note_name)
        time_sec = _music_params.noteLengthToTimeSec(note_length)
        sound_list.append((frequency, time_sec, amplitude))
    return gentones(sound_list)


###########################################################################
# Internal attributes, functions and classes
###########################################################################


class _MusicParams:

    def __init__(self):
        self.__tempo = 120.0
        self.__amplitude = 0.5

    def getTempo(self):
        return self.__tempo

    def setTempo(self, tempo):
        if tempo <= 0.0:
            raise ValueError(
                "Tempo must be greater than zero beats per minute.")
        self.__tempo = tempo

    def getAmplitude(self):
        return self.__amplitude

    def setAmplitude(self, amplitude):
        # we allow negative amplitude as an undocumented feature -
        # it just inverts the resulting output waveform
        if abs(amplitude) > 1.0:
            raise ValueError(
                "Amplitude must be between 0.0 and 1.0, inclusive.")
        self.__amplitude = amplitude

    def noteLengthToTimeSec(self, note_length):
        return (note_length * 4.0 * 60.0) / self.__tempo

_music_params = _MusicParams()

# end-of-file
