# clock.py
# Clock animation example

from cpif.graphics import DrawingWindow

import math
import time


def drawFace(x, y, size):
    """Draw a clock face in the graphics window.
    x, y: coordinates of the center of the clock
    size: distance, in pixels, from the center of the clock face to the 
          outer rim.
    """
    dw.circle(x, y, size, 'black', fill='white', width=2)
    angle = 0.
    for hour in ['12', '1', '2', '3', '4', '5', '6', '7', '8', '9', '10', 
                 '11']:
        # draw number
        xn = x + math.sin(angle * math.pi / 180.) * (size * 0.9)
        yn = y - math.cos(angle * math.pi / 180.) * (size * 0.9)
        dw.drawText(xn, yn, hour, 'black')
        angle = angle + 30.


def drawTime(x, y, size):
    """
    Draw the current time on a clock face.
    x, y: coordinates of the center of the clock
    size: distance, in pixels, from the center of the clock face to the 
          outer rim.
    Returns the ID's of the hand lines: (hour_hand, min_hand, sec_hand)
    """
    hours, minutes, seconds = time.localtime()[3:6]
    hour_angle = (hours + minutes/60.) * (360 / 12.)
    minute_angle = minutes * (360 / 60.)
    second_angle = seconds * (360 / 60.)
    hh = drawHand(x, y, hour_angle, size * 0.5, 5)
    mh = drawHand(x, y, minute_angle, size * 0.7, 2)
    sh = drawHand(x, y, second_angle, size * 0.9, 1)
    return (hh, mh, sh)


def drawHand(x, y, angle, length, width):
    """
    Draw a "hand" on a clock face.
    x, y: coordinates of the center of the clock
    angle: the angle to draw the hand at, in degrees clockwise from 12
    length: the length, in pixels, of the hand
    width: the width, in pixels, of the hand
    """
    xh = x + math.sin(angle * math.pi / 180.) * length
    yh = y - math.cos(angle * math.pi / 180.) * length
    return dw.line(x, y, xh, yh, 'black', width=width)


class Clock:

    def __init__(self, x, y, size):
        drawFace(x, y, size)
        # draw Hour Hand, Minute Hand, Second Hand
        self.hh, self.mh, self.sh = drawTime(x, y, size)
        # save location and size of clock
        self.x = x
        self.y = y
        self.size = size

    def update(self):
        """
        Re-draw the current time, then re-set the timer to do it again.
        x, y: coordinates of the center of the clock
        size: distance, in pixels, from the center of the clock face to the 
              outer rim.
        hour_hand, minute_hand, second_hand: the ID's of the previous drawn
              clock hands.
        """
        # erase the previous clock hands
        dw.delete(self.hh)
        dw.delete(self.mh)
        dw.delete(self.sh)
        # re-draw the clock hands
        self.hh, self.mh, self.sh = drawTime(self.x, self.y, self.size)


# calculate size and position of clock
window_size = 256
clock_size = window_size * 0.45
cx = window_size / 2
cy = window_size / 2

# draw the clock
dw = DrawingWindow(window_size, window_size, 'gray')
dw.title('Clock')
clock = Clock(cx, cy, clock_size)

# update the clock every second
dw.onTimerCall(1.0, clock.update)

# run the clock
dw.run()

# end-of-file
