# cpif/__init__.py
# Created by David Handy  14 August 2003
"""
cpif means "Computer Programming is Fun"

This is the software that accompanies the book
Computer Programming is Fun!, by David Handy (c) 2003-2006

http://www.handysoftware.com/cpif/
"""

# This is the official version string for the cpif package
from cpif.version import __version__


def datadir():
    """
    Return the path to the CPIF data directory. This directory contains
    subdirectories for the shared media and samples files.
    """
    import os
    import cpif.data
    return os.path.dirname(cpif.data.__file__)


def findFile(filename, default_dir=datadir):
    """Search for a file by name.
    
    Search the current directory first, then the program directory, if any,
    then finally the cpif.datadir() directory.  If the default_dir parameter
    is given, that directory is searched last in place of cpif.datadir().
    (default_dir can either be a string or a function returning a string.)
    
    Return a string containing the full path name of the file. If the file
    does not exist, return None."""
    import os
    if os.path.exists(filename):
        return os.path.abspath(filename)
    programdir = _getProgramDir()
    if programdir:
        fullfile = os.path.join(programdir, filename)
        if os.path.isfile(fullfile):
            return os.path.abspath(fullfile)
    if default_dir is None:
        return None
    if callable(default_dir):
        default_dir = default_dir()
    fullfile = os.path.join(default_dir, filename)
    if os.path.isfile(fullfile):
        return os.path.abspath(fullfile)
    # Can't find it anywhere
    return None


def _getProgramDir():
    import os
    import sys
    program = sys.argv[0]
    if program and os.path.isfile(program):
        programdir = os.path.dirname(program)
        if programdir and os.path.isdir(programdir):
            return os.path.abspath(programdir)
    return None


def findFiles(pattern, default_dir=datadir):
    """Works like findFile(), except that the pattern parameter can be
    either a filename or a filename pattern, like "*.wav", where * is the
    wildcard character. Return a list of the matching files found, or an
    empty list if no matches were found. Use findFile() if you need to get
    the full name of one of the files in the returned list."""
    import os
    import fnmatch
    search_dirs = [os.curdir]
    program_dir = _getProgramDir()
    if program_dir:
        search_dirs.append(program_dir)
    if callable(default_dir):
        default_dir = default_dir()
    if default_dir and os.path.isdir(default_dir):
        search_dirs.append(default_dir)
    result = set()
    for search_dir in search_dirs:
        result.update(fnmatch.filter(os.listdir(search_dir), pattern))
    return sorted(result)


def homedir():
    """Return the path to the user's home directory. On Windows,
    returns the path to the user's My Documents directory."""
    import cpif.util.specialdir
    return cpif.util.specialdir.homedir()


def images():
    """Return the path to the CPIF images directory."""
    import os
    return os.path.join(datadir(), 'images')


def inputInt(prompt=""):
    """
    prompt:
        A string to print before waiting for the user to enter something.

    Return the string that the user entered, converted to an integer, or
    None if the string was not a valid integer.
    """
    answer = raw_input(prompt)
    try:
        i = int(answer.strip())
    except ValueError:
        i = None
    return i


def inputNumber(prompt=""):
    """
    prompt:
        A string to print before waiting for the user to enter something.

    Return the string that the user entered, converted to a number, or
    None if the string was not a valid number.
    """
    answer = raw_input(prompt)
    try:
        n = float(answer.strip())
    except ValueError:
        n = None
    return n


def myprograms():
    """Return the path to the user's MyPrograms directory."""
    import cpif.util.specialdir
    return cpif.util.specialdir.myprograms()


def reverse(data):
    """
    Return a string or list with the characters or items in reverse
    order.

    >>> reverse('abc')
    'cba'
    >>> reverse([1, 2, 3])
    [3, 2, 1]
    """
    if isinstance(data, basestring):
        L = [c for c in data]
        L.reverse()
        return ''.join(L)
    else:
        L = list(data)
        L.reverse()
        return L


def setpath(subdir=None):
    r"""
    Change the Python path so that we can import modules from the MyPrograms
    directory. The other modules current in the Python path will remain
    available for import. If this function is called twice, it will make sure
    MyPrograms is not added twice.

    subdir:
        If this parameter is given, then it is the name of a subdirectory
        of the MyPrograms directory to be added to the Python path. For
        example, if your name is Nate and you created a sub-directory of
        MyPrograms called 'nate' to put your modules in, then pass 'nate' as
        this parameter.

        If you pass an absolute path name (i.e. one that begins with C:\) as
        this parameter, then that directory is placed in the Python path.
    """
    import os
    import sys
    if subdir:
        pathdir = os.path.join(myprograms(), subdir)
    else:
        pathdir = myprograms()
    pathdir = os.path.abspath(pathdir)
    pathdir = os.path.normpath(pathdir)
    pathdir = os.path.normcase(pathdir)
    if not pathdir in sys.path:
        sys.path.insert(0, pathdir)


def sounds():
    """Return the path to the CPIF sounds directory."""
    import os
    return os.path.join(datadir(), 'sounds')


def welcome():
    """Run the Welcome Program that copies the student files to the
    user's MyPrograms directory."""
    from cpif.welcomelib import runwelcome
    return runwelcome(datadir())


def test():
    """
    Run a series of basic tests on the cpif package.
    """
    print "cpif self-test"
    import doctest
    import os
    import sys
    doctest.testmod(sys.modules[__name__])
    assert os.path.isdir(datadir())
    assert os.path.isdir(homedir())
    assert os.path.isdir(myprograms())
    assert os.path.isdir(images())
    assert os.path.isdir(sounds())
    # Can't call welcome() as it waits for the user to click a button.
    assert callable(welcome)
    import cpif.welcomelib
    assert callable(cpif.welcomelib.runwelcome)
    assert os.path.isdir(os.path.join(datadir(), 'samples'))
    app = cpif.welcomelib.WelcomeApp(datadir())
    app.master.after(2000, lambda app: app.master.destroy(), app)
    app.mainloop()
    samples_dir = os.path.join(myprograms(), 'samples')
    if not os.path.isdir(samples_dir):
        print "Installing samples"
    else:
        print "Re-installing samples"
    cpif.welcomelib.installFiles()
    def assertSamplesOk():
        assert os.path.isdir(samples_dir)
        import glob
        samples_files = glob.glob(os.path.join(samples_dir, '*.py'))
        assert len(samples_files) >= 8
        assert os.path.isfile(os.path.join(samples_dir, 'README.txt'))
    assertSamplesOk()
    import cpif.graphics
    doctest.testmod(cpif.graphics)
    print "Done"


# end-of-file
