# specialdir.py
# Created by David Handy  16 August 2003
"""Find the paths to special directories, such as the users' home."""

# IMPORTANT! Always keep these module dependencies to a minimum so that the
# setup program built with py2exe can always import this module.
#
# Keep this module backwards-compatible with Python 1.5.2 for setup
# purposes. (Gracefully degrade when _winreg is not present.)

import os
import re
import StringIO
import sys

MY_PROGRAMS_DIR = "MyPrograms"

MY_PROGRAMS_README = """\
cpif = Computer Programming Is Fun!

This MyPrograms directory was created by the cpif software package that
comes with the book "Computer Programming is Fun!" by David Handy (c) 2003,
2004

The MyPrograms directory is a good place to put your own programs and files
that you create. The cpif.myprograms() function is provided to help your
Python program easily locate this directory.

Go to http://www.handysoftware.com/cpif/ to see how you can get your own
copy of the book, and learn computer programming with Python.
"""


class Error(Exception): pass


def homedir():
    """
    Return the path to the user's home directory.
    On Windows this is the same as the user's My Documents directory.
    """
    if sys.platform[:3] == 'win':
        try:
            return _GetMyDocuments()
        except (ImportError, EnvironmentError):
            return _GetMyDocumentsFallback()
    else:
        return _homedir()


def _homedir():
    # Get the user's home directory on non-Windows systems
    # Adapted from idlelib/configHandler.py IdleConf.GetUserCfgDir()
    userDir=os.path.expanduser('~')
    if userDir != '~': #'HOME' exists as a key in os.environ
        if not os.path.exists(userDir):
            warn=('\n Warning: HOME environment variable points to\n '+
                    userDir+'\n but the path does not exist.\n')
            sys.stderr.write(warn)
            userDir='~'
    if userDir=='~': #we still don't have a home directory
        #traditionally idle has defaulted to os.getcwd(), is this adeqate?
        userDir = os.getcwd() #hack for no real homedir
    return userDir


def _GetMyDocuments():
    """
    Get the My Documents directory on Windows.
    Raise EnvironmentError if we can't get the info from the registry.
    Raise Error if the path we got from the registry doesn't exist.
    """
    import _winreg
    subkey = \
       r"Software\Microsoft\Windows\CurrentVersion\Explorer\User Shell Folders"
    key = _winreg.OpenKey(_winreg.HKEY_CURRENT_USER, subkey)
    try:
        path, val_type = _winreg.QueryValueEx(key, "Personal")
    finally:
        key.Close()
    if val_type == _winreg.REG_SZ:
        if not os.path.isdir(path):
            msg = ("My Documents path didn't exist.\n"
                   "path = " + repr(path) + "\n"
                   "val_type = " + repr(val_type))
            sys.stderr.write(msg + '\n')
            raise Error(msg)
        return path
    elif val_type == _winreg.REG_EXPAND_SZ:
        # path may be in the form: u'%USERPROFILE%\\My Documents'
        path2 = re.sub(r'%([^%]+)%', r'${\1}', path)
        path3 = os.path.expandvars(path2)
        if not os.path.isdir(path3):
            msg = ("My Documents path didn't exist.\n"
                   "path = " + repr(path) + "\n"
                   "path2 = " + repr(path2) + "\n"
                   "path3 = " + repr(path3) + "\n"
                   "val_type = " + repr(val_type))
            sys.stderr.write(msg + '\n')
            raise Error(msg)
        return path3
    else:
        msg = ("Unexpected type for 'User Shell Folders\Personal' "
               "registry value.\n"
               "value = " + repr(path) + "\n"
               "val_type = " + repr(val_type))
        raise EnvironmentError(msg)


def _GetMyDocumentsFallback():
    # Attempt to get the MyDocuments directory when _winreg doesn't exist
    # (Python 1.5.2 backwards compatibility.) This is only expected to work
    # on Win9X. It is not likely that someone is running a version of Python
    # that dates back to before their version of Windows was released.
    my_documents = r"C:\My Documents"
    if not os.path.isdir(my_documents):
        # We could reach this point if:
        #
        # 1. We are running Windows NT
        # 2. Python 1.5.2 had been previously installed
        # 3. Someone double-clicked setup.py on the install CD
        #
        # If we don't fail the install at this point, the user will be
        # disappointed later when the updated Python is used to call this
        # function again and it doesn't point to where the user files got
        # installed.
        raise Exception("Cannot find your My Documents directory. Sorry.")
    return my_documents


def myprograms():
    """Return the path to the user's MyPrograms directory.

    This is a subdirectory of the user's home directory. Creates the
    directory if it does not already exist."""
    programs_dir = os.path.join(homedir(), MY_PROGRAMS_DIR)
    if not os.path.isdir(programs_dir):
        try:
            # Create the MyPrograms directory
            os.mkdir(programs_dir)
            # Put the README.txt in MyPrograms
            f = open(os.path.join(programs_dir, 'README.txt'), 'w')
            readme = StringIO.StringIO(MY_PROGRAMS_README)
            while 1:
                line = readme.readline()
                if not line:
                    break
                f.write(line)
            f.close()
            readme.close()
        except (IOError, OSError):
            # If the strategy is good enough for IDLE it's good enough for me
            warn=('\nWarning: unable to create MyPrograms directory\n ' +
                  programs_dir + '\n')
            sys.stderr.write(warn)
    return programs_dir



# end-of-file
