# sound.py
# Copyright (C) 2004 by David Handy
# This is the sound module in the cpif package.
# See http://www.handysoftware.com/cpif/
"""
Sound module.

This module is useful for playing sound files and generating special
effects. It can also be used with the music module to play songs.
"""

import audioop
import copy
import logging
import math
import Queue
import random
import cStringIO
import struct
import sys
import tempfile
import threading
import traceback
import wave

import cpif

log = logging.getLogger('cpif.sound')

DEFAULT_SAMPLE_RATE = 22050

def setdebug():
    if log.getEffectiveLevel() <> logging.INFO:
        log.setLevel(logging.INFO)
        log.addHandler(logging.StreamHandler())


def beep(frequency, time_sec, amplitude=0.5, wait=True):
    """
    Generate and play a tone.

    frequency: The sound frequency in Hz (cycles per second)
    time_sec: The length of time, in seconds, of the sound (floating-point)
    amplitude: The relative loudness, 1.0 is maximum, 0.0 is silent.
               Default is 0.5, or 1/2 of maximum volume.
    wait: If True, don't return until the sound is done playing.
          Default is True.
    """
    s = play(gentone(frequency, time_sec, amplitude))
    if wait:
        s.wait()


def clear():
    r"""
    Remove all waiting sounds from the queue of sounds to be played.

    This function does not affect the sound currently being played, if any.
    """
    _sq.clear()


def play(soundfile):
    """
    Play a WAV format sound file on the PC's main sound card.

    soundfile:
        If soundfile is a string, it is used as the name of a WAV file. If
        the file name does not contain a directory path, the current
        directory, the program directory, and the cpif.sounds() directory
        are searched for the file name.

        This parameter can also be a file object in WAV format, or a 
        Sound object as returned by gentone().

    This function returns immediately, and the sound starts playing on the
    PC's main sound card. If this function is called again before the first
    sound is done playing, the second sound will start playing when the
    first sound is complete. You can call play() multiple times with the
    same soundfile, and it will repeat the sound.

    Return the Sound object that is to be played.

    >>> from cpif import sound
    >>> soundobj = sound.play('hello.wav')
    >>> soundobj.wait() # wait till the sound is done playing
    """
    if not hasattr(soundfile, 'play'):
        soundfile = Sound(soundfile)
    soundfile.play()
    return soundfile


def wait():
    r"""
    Wait until all sounds have completed playing.
    Return immediately if no sounds are currently playing.
    """
    _sq.wait()


def getqueuetime():
    """
    Return the total time, in seconds, of all sound files currently
    waiting to be played. Return 0.0 if no sounds are currently waiting.
    """
    return _sq.getqueuetime()


def gensilence(time_sec, outfile=None):
    """
    Generate a Sound object consisting of time_sec seconds of silence.

    This is really only useful for putting space between sounds that you are
    combining using mix() and join().
    """
    wavef = _getOutFileObject(outfile)
    waveout = _initWave(wavef)
    samplerate = waveout.getframerate()
    samplegen = _GenSilence(samplerate, time_sec)
    for block in samplegen:
        waveout.writeframesraw(block)
    waveout.close()
    wavef.seek(0)
    return Sound(wavef)


def gentone(frequency, time_sec, amplitude=0.5, phase=None, outfile=None,
            sample_rate=DEFAULT_SAMPLE_RATE):
    """
    Generate a sound.

    frequency: The sound frequency in Hz (cycles per second)
    time_sec: The length of time, in seconds, of the sound (floating-point)
    amplitude: The relative loudness, 1.0 is maximum, 0.0 is silent.
    phase: The starting phase angle in degrees. Default is a random phase
           angle (recommended.)
    outfile: (optional) the file object to put the resulting sound in.
             Useful for creating large sound files on the hard drive.
    sample_rate: (optional) Set the sample rate at which the sound card will
             operate, in samples per second. Recommended: Keep the default of
             22050 samples per second.

    Return a Sound object compatible with the play() function.

    >>> # Generate and play a short, high-pitched beep
    >>> import cpif.sound
    >>> soundobj = cpif.sound.gentone(2000, 0.25, 0.5)
    >>> s = cpif.sound.play(soundobj)
    >>> s is soundobj
    True
    >>> s.wait()
    """
    wavef = _getOutFileObject(outfile)
    waveout = _initWave(wavef, sample_rate=sample_rate)
    _gentone(waveout, frequency, time_sec, amplitude, phase)
    waveout.close()
    wavef.seek(0)
    return Sound(wavef)


def gentones(sound_list, outfile=None, sample_rate=DEFAULT_SAMPLE_RATE):
    """
    Generate a sound from a list of tones.

    sound_list:
        A list tuples containing sound parameters. Each tuple is in the
        format (frequency, time_sec, amplitude), which have the same meaning
        as the parameters of the same name in the gentone() function.
    outfile: (optional) the file object to put the resulting sound in.
             Useful for creating large sound files on the hard drive.
    sample_rate: (optional) Set the sample rate at which the sound card will
             operate, in samples per second. Recommended: Keep the default of
             22050 samples per second.

    Return a Sound object compatible with the play() function.

    >>> # Play three notes separated by short rests
    >>> import cpif.sound
    >>> soundobj = cpif.sound.gentones([
    ...     (261.63, 0.25, 0.5), (None, 0.125, 0.0), # C
    ...     (329.63, 0.25, 0.5), (None, 0.125, 0.0), # E
    ...     (392.00, 0.25, 0.5), (None, 0.125, 0.0), # G
    ...     ])
    >>> cpif.sound.play(soundobj).wait()
    """
    wavef = _getOutFileObject(outfile)
    waveout = _initWave(wavef, sample_rate=sample_rate)
    for frequency, time_sec, amplitude in sound_list:
        _gentone(waveout, frequency, time_sec, amplitude)
    waveout.close()
    wavef.seek(0)
    return Sound(wavef)


def join(sound_list, outfile=None):
    """
    Join each sound in sound_list to the end of the previous sound, and
    return the combined sound.

    The sounds are read but not modified. Do not call this function while
    any of the input sounds are playing, because reading the sound file
    objects at the same time as they are being played causes errors.

    sound_list:
        Sound file names, or file objects in WAV format, or Sound objects as
        returned by gentone().
    outfile: (optional) the file object to put the resulting sound in.
             Useful for creating large sound files on the hard drive.

    Return a Sound object compatible with the play() function.

    >>> # Generate 2 falling tones and splice them into one sound
    >>> import cpif.sound
    >>> s1 = cpif.sound.gentone(392.00, 0.25, 0.5) # G
    >>> s2 = cpif.sound.gentone(329.63, 0.25, 0.5) # E
    >>> s3 = cpif.sound.join([s1, s2])
    >>> cpif.sound.play(s3).wait()
    >>> total_time = s3.gettime()
    >>> abs( s1.gettime() + s2.gettime() - total_time ) < 0.01
    True
    """
    wavein_list, wavef, waveout = _getSoundOpParams(sound_list, outfile)
    for wavein in wavein_list:
        while True:
            data = wavein.readframes(_SAMPLE_BLOCK_SIZE)
            if not data:
                break
            waveout.writeframesraw(data)
        wavein.rewind()
    waveout.close()
    wavef.seek(0)
    return Sound(wavef)


def mix(sound_list, outfile=None):
    """
    Mix all the sounds in sound_list and return the mixed sound object.
    Consider using mul() to reduce the volume of the input sounds before
    mixing them together, or the resulting sound may be saturated and have
    distortion. (However some people like distortion...)

    The sounds are read but not modified. Do not call this function while
    any of the input sounds are playing, because reading the sound file
    objects at the same time as they are being played causes errors.

    sound_list:
        Sound file names, or file objects in WAV format, or Sound objects as
        returned by gentone().
    outfile: (optional) the file object to put the resulting sound in.
             Useful for creating large sound files on the hard drive.

    Return a Sound object compatible with the play() function.
    """
    wavein_list, wavef, waveout = _getSoundOpParams(sound_list, outfile)
    wavein1 = wavein_list[0]
    nchannels = wavein1.getnchannels()
    sampwidth = wavein1.getsampwidth()
    framerate = wavein1.getframerate()
    while True:
        data_list = []
        for wavein in wavein_list:
            data_list.append(wavein.readframes(_SAMPLE_BLOCK_SIZE))
        longest_data_len = max(map(len, data_list))
        if not longest_data_len:
            break
        accum_data = ''
        for data in data_list:
            diff_len = longest_data_len - len(data)
            if diff_len > 0:
                zeros = struct.pack("%dB" % diff_len, *([0] * diff_len))
                data = "%s%s" % (data, zeros)
                #print >> sys.stderr, "padding with", diff_len, "zero bytes"
            if not accum_data:
                accum_data = data
            else:
                try:
                    accum_data = audioop.add(accum_data, data, sampwidth)
                except audioop.error:
                    print >> sys.stderr, "longest_data_len", longest_data_len
                    print >> sys.stderr, "len(accum_data)", len(accum_data)
                    print >> sys.stderr, "len(data)", len(data)
                    raise
        waveout.writeframesraw(accum_data)
    for wavein in wavein_list:
        wavein.rewind()
    waveout.close()
    wavef.seek(0)
    return Sound(wavef)


def mul(soundfile, factor, outfile=None):
    """
    Multiply a sound by a factor. A factor greater than 1.0 makes it
    louder, a factor between 0.0 and less than 1.0 makes it softer.
    """
    if not isinstance(soundfile, Sound):
        soundfile = Sound(soundfile)
    wavein = soundfile.getwavefile()
    wavef = _getOutFileObject(outfile)
    waveout = _initWave(wavef)
    width = wavein.getsampwidth()
    wavein.rewind()
    while True:
        data = wavein.readframes(_SAMPLE_BLOCK_SIZE)
        if not data:
            break
        waveout.writeframesraw(audioop.mul(data, width, factor))
    wavein.rewind()
    waveout.close()
    wavef.seek(0)
    return Sound(wavef)


def findSoundFile(filename):
    """Search for a sound file by name. Search the current directory first,
    then the program directory, if any, then the cpif.sounds() directory.
    Return the full path to the file. If the file does not exist, return
    None."""
    return cpif.findFile(filename, default_dir=cpif.sounds)


def listSoundFiles():
    """Return a list of all the sound files available in the current
    directory, the directory of the current program (if any), and the
    cpif.sounds() directory."""
    return cpif.findFiles('*.wav', default_dir=cpif.sounds)


class Sound:
    """
    A sound object, compatible with the play() function in this module.
    """

    def __init__(self, soundfile):
        """
        Initialize a Sound.

        soundfile:
            A string contining the name of a sound file in WAV format,
            or a file object containing a sound in WAV format.
        """
        self.__soundfile_param = soundfile
        if isinstance(soundfile, basestring):
            fullfile = findSoundFile(soundfile)
            if not fullfile:
                raise IOError("File not found: %s" % soundfile)
            wavef = file(fullfile, 'rb')
        else:
            wavef = soundfile
        wavefile = wave.open(wavef, 'rb')
        time_sec = float(wavefile.getnframes()) / wavefile.getframerate()
        self.__wavef = wavef
        self.__wavefile = wavefile
        self.__time = time_sec
        self.__event = threading.Event()
        self.__event.set() # cleared until play() is called

    def __repr__(self):
        return ("Sound(%s)" % repr(self.__soundfile_param))

    def __str__(self):
        return ("<Sound(%s): "
            "%f sec. "
            "nchannels=%d "
            "sampwidth=%d "
            "framerate=%d "
            "nframes=%d>" %
            (self.__soundfile_param,
             self.__time,
             self.__wavefile.getnchannels(),
             self.__wavefile.getsampwidth(),
             self.__wavefile.getframerate(),
             self.__wavefile.getnframes()))

    def gettime(self):
        """
        Return the length of the sound in seconds.
        """
        return self.__time

    def getwavef(self):
        """
        Get the raw file object containing the WAV sound.
        """
        return self.__wavef

    def getwavefile(self):
        """
        Get the processed WAV data, as returned by wave.open(wavef, 'rb').
        See the wave module documentation for more details.
        """
        return self.__wavefile

    def isPlaying(self):
        """
        Return True iff the sound is currently playing or waiting in the
        play queue.
        """
        return not self.__event.isSet()

    def play(self):
        """
        Start playing this Sound.
        """
        self.__event.clear()
        _sq.play(self, callback=self._donePlaying)

    def _donePlaying(self):
        self.__event.set()

    def close(self):
        """
        We're done using this Sound object.
        """
        if self.__wavefile:
            self.__wavefile.close()
            self.__wavefile = None
        if self.__wavef:
            if self.__wavef is not self.__soundfile_param:
                self.__wavef.close()
            self.__wavef = None
        self.__time = None

    def wait(self, timeout=None):
        """
        Wait until this sound is done playing.  Returns immediately if this
        sound is not being played or is not in the sound queue.
        """
        self.__event.wait(timeout)

    def save(self, outfile):
        """
        Write the sound to another file.
        Don't call this when the sound is playing.

        outfile: A file name or file-like object.
        """
        if isinstance(outfile, basestring):
            outf = file(outfile, 'wb')
        elif hasattr(outfile, 'write') and hasattr(outfile, 'seek'):
            outf = outfile
        else:
            raise ValueError(
                "outfile parameter is neither a filename nor a file-like "
                "object with write() and seek().")
        try:
            # I'm not using the brain-dead implementation below because
            # I am wary of seeking outside of the chunk boundary.
            #self.__wavef.seek(0)
            #shutil.copyfileobj(self.__wavef, outf)
            # Here's the longer, safer implementation
            wavein = self.__wavefile
            waveout = wave.open(outf, 'wb')
            waveout.setnchannels(wavein.getnchannels())
            waveout.setsampwidth(wavein.getsampwidth())
            waveout.setframerate(wavein.getframerate())
            wavein.rewind()
            while True:
                data = wavein.readframes(_SAMPLE_BLOCK_SIZE)
                if not data:
                    break
                waveout.writeframesraw(data)
            waveout.close()
        finally:
            if outf is not outfile:
                outf.close()


###########################################################################
# Internal attributes, functions and classes
###########################################################################


_SAMPLE_BLOCK_SIZE = 4410


def _getOutFileObject(outfile):
    """
    This is definitely an internal function.

    outfile: The file object to put the resulting sound in.
             Useful for creating large sound files on the hard drive.
    
    If outfile is None, create a temporary file-like object for writing and
    reading.
    If outfile is a string, open a disk file of that name for writing and
    reading.
    Otherwise, return outfile.
    """
    # The wave module will fail to write or read frames from or to a
    # non-disk file object on big-endian systems.
    #
    # Why? Because it uses the array module to byte-swap blocks of samples,
    # and it uses array.array.tofile() and .fromfile(),  which require
    # actual disk files and not file-like objects, or else they throw
    # exceptions (I tried cStringIO.StringIO and it failed.)
    #
    # My solution is to use a temporary disk file instead of a
    # cString.StringIO object on big-endian systems only, which the tofile()
    # and fromfile() methods will accept. The temporary disk file is
    # automatically deleted when the file object is closed or
    # garbage-collected.
    #
    if not outfile:
        if sys.byteorder == 'little':
            return cStringIO.StringIO()
        else:
            return tempfile.TemporaryFile()
    else:
        if isinstance(outfile, basestring):
            return file(outfile, 'wb+')
        else:
            return outfile


def _initWave(wavef, sample_rate=DEFAULT_SAMPLE_RATE):
    """
    Internal function to initialize a synthesized wave file object.
    """
    waveout = wave.open(wavef, 'wb')
    waveout.setnchannels(1) # 1 channel = mono
    waveout.setsampwidth(2) # bytes
    waveout.setframerate(sample_rate)
    return waveout


def _gentone(waveout, frequency, time_sec, amplitude, phase=None):
    """
    Internal function.
    """
    samplerate = waveout.getframerate()
    if frequency and amplitude:
        if time_sec < 0.0:
            raise ValueError("time_sec must not be less than zero")
        if abs(amplitude) > 1.0:
            raise ValueError("amplitude must be from 0.0 to 1.0")
        if phase is None:
            # generate random phase angle
            phase = random.uniform(0.0, 360.0)
        samplegen = _GenSineWave(
            samplerate, frequency, time_sec, amplitude, phase)
    else:
        # silence
        samplegen = _GenSilence(samplerate, time_sec)
    for block in samplegen:
        waveout.writeframesraw(block)
    waveout.writeframes('') # sync frame count


def _getSoundOpParams(sound_list, outfile):
    wavein_list = []
    for soundobj in sound_list:
        if not isinstance(soundobj, Sound):
            soundobj = Sound(soundobj)
        wavein = soundobj.getwavefile()
        wavein.rewind()
        wavein_list.append(wavein)
    if not wavein_list:
        raise ValueError("No input sound parameters.")
    wavein_list = _makeCompatibleWaves(wavein_list)
    wavein1 = wavein_list[0]
    wavef = _getOutFileObject(outfile)
    waveout = wave.open(wavef, 'wb')
    waveout.setsampwidth(wavein1.getsampwidth())
    waveout.setnchannels(wavein1.getnchannels())
    waveout.setframerate(wavein1.getframerate())
    return (wavein_list, wavef, waveout)


def _makeCompatibleWaves(wavein_list):
    """
    Take a sequence of sounds and return a sequence of the same sounds,
    all converted to a compatible sample rate and format. Each sound is a
    wave object as returned by wave.open(wavef, 'rb').

    If all of the input sounds have the same sample rate and format, the
    list of sounds is returned unchanged. Otherwise, each sound is converted
    to the frame rate of the sound with the highest frame rate, the sample
    width of the sound with the widest sample width, and the format of the
    sound with the widest format (stero vs. mono.)
    """
    max_nchannels = 0
    max_sampwidth = 0
    max_framerate = 0
    for wavein in wavein_list:
        if wavein.getnchannels() > max_nchannels:
            max_nchannels = wavein.getnchannels()
        if wavein.getsampwidth() > max_sampwidth:
            max_sampwidth = wavein.getsampwidth()
        if wavein.getframerate() > max_framerate:
            max_framerate = wavein.getframerate()
    result = []
    for wavein in wavein_list:
        if (wavein.getnchannels() == max_nchannels and
            wavein.getsampwidth() == max_sampwidth and
            wavein.getframerate() == max_framerate):
            result.append(wavein)
        else:
            result.append(_convertWave(wavein, max_nchannels, max_sampwidth,
                                   max_framerate))
    return result


def _convertWave(wavein, nchannels, sampwidth, framerate, outfile=None):
    """
    Convert a wave file to the desired format and sample rate.
    
    wavein: a sound in wave format, as returned by wave.open(wavef, 'rb')
    nchannels, sampwidth, framerate:
        The desired number of channels, sample width, and sample rate.
    outfile: The backing output file object. If not given, the backing file
             object is either a cStringIO object or a temporary file that
             will be deleted when the file is closed.
    Return a sound in wave format, as returned by wave.open(wavef, 'rb').
    """
    wavef = _getOutFileObject(outfile)
    waveout = wave.open(wavef, 'wb')
    waveout.setnchannels(nchannels)
    waveout.setsampwidth(sampwidth)
    waveout.setframerate(framerate)
    in_framerate = wavein.getframerate()
    in_sampwidth = wavein.getsampwidth()
    in_nchannels = wavein.getnchannels()
    wavein.rewind()
    state = None
    while True:
        data = wavein.readframes(_SAMPLE_BLOCK_SIZE)
        if in_framerate <> framerate:
            data, state = audioop.ratecv(data, in_sampwidth, in_nchannels,
                                  in_framerate, framerate, state)
        if in_sampwidth <> sampwidth:
            data = audioop.lin2lin(data, in_sampwidth, sampwidth)
        if in_nchannels <> nchannels:
            if in_nchannels == 1 and nchannels == 2:
                data = audioop.tostereo(data, sampwidth, 1.0, 1.0)
            elif in_channels == 2 and nchannels == 1:
                data = audioop.tomono(data, sampwidth, 0.5, 0.5) # ???
            else:
                raise ValueError(
                    "Can't cope with " + str(in_nchannels) +
                    "input channel(s) and " + str(nchannels) +
                    "output channel(s).")
        waveout.writeframesraw(data)
    waveout.close()
    wavef.seek(0)
    return wave.open(wavef, 'rb')


def _GenSilence(samplerate, time_sec):
    """
    Silence generator, for putting spaces between tones.

    Generate audio samples in signed 16-bit Mono PCM format, native byte
    order. Yield strings containing blocks of samples ready for output or
    for putting in an audio file (the wave module handles byte-swapping).
    
    samplerate is in frames (samples) per second
    time_sec is a floating point number, in seconds
    """
    total_samples = int(time_sec * samplerate)
    num_blocks = total_samples // _SAMPLE_BLOCK_SIZE
    leftover_samples = total_samples % _SAMPLE_BLOCK_SIZE
    block = struct.pack("%dh" % _SAMPLE_BLOCK_SIZE, 
        *([0] * _SAMPLE_BLOCK_SIZE))
    # yield the sample blocks
    for i in xrange(num_blocks):
        yield block
    # yield the last partial block
    if leftover_samples:
        yield block[:(leftover_samples*2)]


def _GenSineWave(samplerate, frequency, time_sec, amplitude, phase):
    """
    Sine wave generator.

    Generate audio samples in signed 16-bit Mono PCM format, native byte
    order. Yield strings containing blocks of samples ready for output or
    for putting in an audio file (the wave module handles byte-swapping).
    
    frequency is in Hz
    time_sec is a floating point number, in seconds
    samplerate is in frames (samples) per second
    amplitude is between 0 and 1.0
    phase is between 0.0 and 360.0 degrees
    """
    import __builtin__
    float = __builtin__.float # turn global into local for speed
    frequency = float(frequency)
    samplerate = float(samplerate)
    phase = phase * (math.pi / 180.0) # convert to radians
    # total number of samples to play
    n = int(time_sec * samplerate)
    min_block_samples = _SAMPLE_BLOCK_SIZE
    cycles_per_block = \
        float(int((frequency/samplerate)*min_block_samples) + 1)
    samples_per_block = int((cycles_per_block / frequency) * samplerate)
    num_blocks = int(n / samples_per_block)
    leftover_samples = n % samples_per_block
    # generate one block of samples
    sin = math.sin
    pi2f = math.pi * 2.0 * frequency
    amp = 32767.0 * amplitude # 16-bit dynamic range
    pack = struct.pack
    blockStream = cStringIO.StringIO()
    write = blockStream.write
    for i in xrange(samples_per_block):
        y = amp * sin(((pi2f * float(i)) / samplerate) + phase)
        write(pack("h", int(y))) # "<h" to force little-endian byte order
    block = blockStream.getvalue()
    # yield the sample blocks
    for i in xrange(num_blocks):
        yield block
    # yield the last partial block
    if leftover_samples:
        yield block[:(leftover_samples*2)]


class _SoundQueue:
    """
    This class implements a queue for playing sounds. Do not use this
    class directly; call the public sound module functions instead.
    """

    def __init__(self):
        self.__queue = Queue.Queue()
        self.__thread_lock = threading.RLock()
        self.__thread = None
        self.__empty_event = threading.Event()
        self.__empty_event.set() # queue is initially empty
        # this general lock controls access to the remaining attributes
        self.__lock = threading.RLock()
        self.__sound_id = 0
        self.__map = {} # {sound_id: soundfile}
        self.__qtime = 0.0
        self.player = None

    def play(self, soundfile, callback=None):
        assert isinstance(soundfile, Sound)
        self._add(soundfile, callback)
        self._startThread()

    def clear(self):
        self._clear()

    def wait(self, timeout=None):
        self.__empty_event.wait(timeout)

    def getqueuetime(self):
        return self.__qtime

    def _add(self, soundfile, callback):
        self.__lock.acquire()
        try:
            self.__sound_id += 1
            sound_id = self.__sound_id
            self.__map[sound_id] = (soundfile, callback)
            self.__qtime += soundfile.gettime()
        finally:
            self.__lock.release()
        self.__empty_event.clear()
        self.__queue.put(sound_id)
        return sound_id

    def _remove(self, sound_id):
        self.__lock.acquire()
        try:
            soundfile, callback = self.__map.pop(sound_id, (None, None))
            if soundfile:
                time_sec = soundfile.gettime()
            else:
                time_sec = 0.0
            self.__qtime -= time_sec
        finally:
            self.__lock.release()
        return (soundfile, callback)

    def _clear(self):
        self.__lock.acquire()
        try:
            sound_map = copy.copy(self.__map)
            self.__map = {}
            self.__qtime = 0.0
        finally:
            self.__lock.release()
        # stop the currently playing sound, if possible
        if self.player:
            if hasattr(self.player, 'stop'):
                self.player.stop()
        # notify any threads waiting on these objects
        for soundfile, callback in sound_map.values():
            if callable(callback):
                callback() # this had better be quick

    def _startThread(self):
        # start the Thread if it is not already running
        # the lock is to prevent starting two threads
        self.__thread_lock.acquire()
        try:
            if not self.__thread:
                self.__thread = threading.Thread(target=self._run)
                self.__thread.setDaemon(True)
                self.__thread.start()
        finally:
            self.__thread_lock.release()

    def _run(self):
        # run the sound queue
        try:
            self.player = _openplayer()
            try:
                block = True
                while True:
                    try:
                        sound_id = self.__queue.get(block=block)
                    except Queue.Empty:
                        self.__empty_event.set()
                        block = True
                        continue
                    block = False
                    soundfile, callback = self._remove(sound_id)
                    if not soundfile:
                        # sound must have been removed
                        continue
                    try:
                        try:
                            self.player.play(soundfile) # blocks
                        except:
                            traceback.print_exc()
                            raise
                    finally:
                        if callable(callback):
                            callback() # this had better be quick
            finally:
                try:
                    self.player.close()
                except:
                    traceback.print_exc()
                self.player = None
        finally:
            self.__thread = None
            self.__empty_event.set()
            try:
                self._clear()
            except:
                traceback.print_exc()

# The singleton _SoundQueue instance
_sq = _SoundQueue()


class _PlayerLinux:
    """
    Sound player for Linux and any other system with ossaudiodev.
    """

    def __init__(self):
        import ossaudiodev
        self.__audiomod = ossaudiodev
        self.__audiodev = None
        self.__format = None
        self.__nchannels = None
        self.__samplerate = None
        # Platform-specific, not all support stopping sound
        self.__stop = False

    def play(self, wavefile):
        """
        wavefile is an object created by or compatible with the object
        returned by wave.open(fp, 'rb'), or an instance with a getwavefile()
        method that returns such an object.
        """
        if hasattr(wavefile, 'getwavefile'):
            wavefile = wavefile.getwavefile()
        self._setCompatibleMode(wavefile)
        bufsize = self.__audiodev.bufsize()
        wavefile.rewind()
        self.__stop = False
        while not self.__stop:
            data = wavefile.readframes(bufsize)
            if not data:
                break
            self.__audiodev.write(data)
        self.__audiodev.sync()

    def stop(self):
        self.__stop = True

    def _setCompatibleMode(self, wavefile):
        """
        Ensure that the sound device is open and set to the right mode.
        """
        if wavefile.getsampwidth() == 1:
            log.info("wavefile has 8-bit unsigned samples")
            req_format = self.__audiomod.AFMT_U8
        else:
            # The wave module tries to be helpful by returning 16-bit
            # samples converted to native byte order, whether we want them
            # that way or not. Therefore we'll try and open the audio device
            # in 16-bit signed native-byte order mode, and hope it works.
            # (I don't have any big-endian machines to try this on.)
            if sys.byteorder == 'little':
                log.info("wavefile has signed 16-bit little-endian samples")
                req_format = self.__audiomod.AFMT_S16_LE
            else:
                log.info("wavefile signed 16-bit big-endian samples")
                req_format = self.__audiomod.AFMT_S16_BE
        req_nchannels = wavefile.getnchannels()
        req_samplerate = wavefile.getframerate()
        log.info("wavefile channels: %r  samplerate: %r" % 
                 (req_nchannels, req_samplerate))
        if (self.__audiodev and
                self.__format == req_format and
                self.__nchannels == req_nchannels and
                self.__samplerate == req_samplerate):
            # no need to change anything
            log.info("Assuming audiodev open and set to correct mode")
            return
        self.close()
        log.info("Opening audiodev")
        self.__audiodev = self.__audiomod.open('w')
        # Is this check helpful?
        # Might the sound hardware convert from the requested format to
        # its native format? I'll just try setting the parameters and
        # let the device tell us if it doesn't like them.
        #if not (self.__audiodev.getfmts() & req_format):
        #    self.close()
        #    raise IOError("Sound device can't handle requested format.")
        log.info("Setting audiodev format to %r" % req_format)
        self.__format = self.__audiodev.setfmt(req_format)
        log.info("Setting audiodev channels to %r" % req_nchannels)
        self.__nchannels = self.__audiodev.channels(req_nchannels)
        log.info("Setting audiodev speed to %r" % req_samplerate)
        samplerate = self.__samplerate = self.__audiodev.speed(req_samplerate)
        log.info("Actual samplerate: %r" % samplerate)

    def close(self):
        """
        Done with this Player object.
        """
        if self.__audiodev:
            try:
                self.__audiodev.close()
            except IOError:
                pass
            self.__audiodev = None


class _PlayerWindows:
    """
    Sound player for Windows, uses winsound module.
    """

    def __init__(self):
        import winsound
        self.__audiomod = winsound

    def play(self, wavefile):
        """
        wavefile is an object created by or compatible with the object
        returned by wave.open(fp, 'rb'), or an instance with a getwavefile()
        method that returns such an object.
        """
        if hasattr(wavefile, 'getwavefile'):
            wavefile = wavefile.getwavefile()
        wavef = wavefile.getfp()
        if hasattr(wavef, 'file'):
            wavef = wavef.file
        flags = self.__audiomod.SND_NODEFAULT
        if isinstance(wavef, file):
            soundfile = wavef.name
            flags = flags | self.__audiomod.SND_FILENAME
        else:
            if hasattr(wavef, 'getvalue'):
                soundfile = wavef.getvalue()
            else:
                wavef.seek(0)
                soundfile = wavef.read()
            flags = flags | self.__audiomod.SND_MEMORY
        self.__audiomod.PlaySound(soundfile, flags)

    def close(self):
        """
        Done with this Player object.
        """
        pass


def _openplayer():
    """
    Return the Player object to use for sound playback functions.

    The Player object will have the following methods:
        play(wavefile) - wavefile as created by wave.open(wavef, 'rb')
        close() - use when done
    """
    try:
        import ossaudiodev
        return _PlayerLinux()
    except ImportError:
        pass
    try:
        import winsound
        return _PlayerWindows()
    except ImportError:
        pass
    raise Exception(
        "Could not find sound support for this type of computer.")


__test__ = {
    'clear_getqueutime': r"""
    >>> import sys, time
    >>> import cpif.sound
    >>> cpif.sound.wait()
    >>> # Stack up sounds in the queue, then check the queue time before
    >>> # and after clearing the queue.
    >>> def tryit():
    ...     s1 = cpif.sound.gentone(220, 1.5, 0.5)
    ...     s2 = cpif.sound.gentone(440, 1.5, 0.5)
    ...     s3 = cpif.sound.gentone(880, 1.5, 0.5)
    ...     for soundobj in (s1, s2, s3):
    ...         dummy = cpif.sound.play(soundobj)
    ...     # Delay to allow s1 to begin playing
    ...     time.sleep(0.001)
    ...     t1 = cpif.sound.getqueuetime()
    ...     cpif.sound.clear()
    ...     t2 = cpif.sound.getqueuetime()
    ...     return (t1, t2)
    ... 
    >>> # You should hear 1 or 2 tones, depending your system and soundcard
    >>> qt1, qt2 = tryit()
    >>> sys.stderr.write("queuetime before=%s\n" % str(qt1))
    >>> sys.stderr.write("queuetime after =%s\n" % str(qt2))
    >>> print qt1 > 1.49
    True
    >>> print qt2 == 0.0
    True
    """,
    'wait_timing': r"""
    >>> import cpif.sound
    >>> import time
    >>> # wait for any other sounds to complete
    >>> cpif.sound.wait()
    >>> # play a short beep
    >>> # wait for the beep to complete, then check the delay
    >>> time_sec = 3.0
    >>> delta = time_sec * 0.25
    >>> soundobj = cpif.sound.gentone(1000, time_sec, 0.5)
    >>> def tryit():
    ...     t1 = time.time()
    ...     dummy = cpif.sound.play(soundobj)
    ...     cpif.sound.wait()
    ...     t2 = time.time()
    ...     return t2 - t1
    ... 
    >>> actual = tryit()
    >>> print abs( actual - time_sec ) < delta
    True
    >>> # Diagnostic output
    >>> import sys
    >>> sys.stderr.write("tone time  =%s\n" % str(time_sec))
    >>> sys.stderr.write("actual time=%s\n" % str(actual))
    """,
    }


def main():
    if len(sys.argv) < 2:
        print "Play sound file"
        print "use: python cpif/sound.py <wave-file-name>"
        return
    soundfile = sys.argv[1]
    print "Playing wave file:", soundfile
    soundobj = play(soundfile)
    soundobj.wait() # wait till the sound is done playing


if __name__ == '__main__':
    main()

# end-of-file
