# graphics.py
# Copyright (C) 2003-2009 by David Handy
# See the LICENSE.txt file for terms and conditions of use.
"""
Graphics module

Use the DrawingWindow class to create a window for graphics purposes.
"""

import inspect
import math
import os
import sys
import threading
import time
from tkFont import Font
import Tkinter as Tk
import tkMessageBox

import cpif


def centerWindow(window):
    """
    Center a top-level Tk window on the screen.
    Make sure that window pops up above all other windows.
    """
    w, h = window.winfo_width(), window.winfo_height()
    if (w, h) == (1, 1):
        # geometry manager has not figured out width and height yet
        window.update()
        w, h = window.winfo_reqwidth(), window.winfo_reqheight()
    sw, sh = window.winfo_screenwidth(), window.winfo_screenheight()
    new_geometry = "=%dx%d+%d+%d" % (w, h, (sw/2 - w/2), (sh/2 - h/2))
    window.wm_geometry(new_geometry)
    window.wm_deiconify() # MS Windows: raises the window above others
    window.lift() # Linux: raises the window above others


def findImageFile(filename):
    """Search for an image file by name. Search the current directory first,
    then the program directory, if any, then the cpif.images() directory.
    Return the full path to the file. If the file does not exist, return
    None."""
    return cpif.findFile(filename, default_dir=cpif.images)


SUPPORTED_IMAGE_FILE_EXTENSIONS = ['.gif', '.bmp']

def listImageFiles():
    """Return a list of all the image files in the formats supported by
    DrawingWindow.drawImage() (.gif, .bmp) available in the current
    directory, the directory of the current program (if any), and the
    cpif.images() directory."""
    result = set()
    for ext in SUPPORTED_IMAGE_FILE_EXTENSIONS:
        result.update(cpif.findFiles('*'+ext, default_dir=cpif.images))
    return sorted(result)


def pythonIsInteractive():
    """
    Return True if it appears that Python is running "interactively",
    meaning running from the ">>> " prompt in the IDLE Python Shell window
    or in the regular Python command-line console.  Return False otherwise.
    """
    if hasattr(sys, 'ps1'):
        # Regular Python command-line console
        is_interactive = True
    elif (not sys.argv) or (not sys.argv[0]):
        # No script name - this works even with IDLE
        is_interactive = True
    else:
        is_interactive = False
    return is_interactive


class DrawingWindow(object):
    """
    A window containing a drawing canvas, useful for drawing, animation,
    etc.
    """

    # constants for arc() - style of drawing arc
    PIESLICE = Tk.PIESLICE
    CHORD = Tk.CHORD
    ARC = Tk.ARC

    # Font sizes
    HUGE = 22
    LARGE = 15
    MEDIUM = 10
    SMALL = 6

    # Textentry border sizes in pixels
    TEXTENTRY_PADX = 4.0
    TEXTENTRY_PADY = 4.0

    def __init__(self, width, height, background='white',
                 startup_callback=None, window=None,
                 interactive=None):
        """
        width, height: the size of the drawing canvas, in pixels

        background: (optional) background color of the drawing canvas,
                default is 'white'.

        startup_callback: (optional) function to call when the DrawingWindow 
                is displayed for the first time. This function or method is a
                good place to put code to draw the initial contents.  It will
                be called with no parameters.  None or default means no
                function will be called.

        window: (optional) parent window, None or default means make a new
                parent window.

        interactive: (optional) True to indicate that this DrawingWindow is
                being created from a Python command prompt, False to
                indicate this DrawingWindow is being run as part of a
                program, None or default to determine interactive status
                automatically.
        """
        self.width = width
        self.height = height
        self.background = background
        if startup_callback:
            _checkParams(startup_callback, 0)
        self.__startup_callback = startup_callback
        self.__window_param = window
        self.interactive = interactive
        if self.interactive is None:
            self.interactive = pythonIsInteractive()
        ###
        self.__title = 'Drawing Window'
        self.__tag_seq_num = 0
        self.__tag_stack = ['__ALL__']
        self.__resetItemData()
        self.__resize_callback = None
        self.__delete_callback = None
        self.__size_font_map = {}
        self.__last_frame_num = -1
        self.__first_configure = True
        self.__initTk()

    def __initTk(self):
        # Create the window and canvas
        background = _convertColor(self.background)
        if not self.__window_param:
            self.window = Tk.Tk()
        else:
            self.window = self.__window_param
        # XXX sometimes causes crashes on ActivePython on Linux
        self.window.wm_protocol("WM_DELETE_WINDOW", self._onDeleteWindow)
        self.window.title(self.__title)
        self.canvas = Tk.Canvas(self.window, 
                                width=self.width, 
                                height=self.height,
                                background=background,
                                highlightthickness=0,
                                borderwidth=0)
        self.canvas.pack(expand=True, fill=Tk.BOTH)
        # The following line, if uncommented, would make the window
        # non-resizable in either direction.
        #self.window.resizable(0, 0)
        # Handle window resize events
        self.canvas.bind('<Configure>', self._onConfigure)
        # Create the fonts
        self.__size_font_map = self._createSizeFontMap()
        # Make the drawing commands take effect
        if self.interactive:
            self.window.update()

    def __cleanupItemData(self, id):
        # This method must be called when deleting anything that might
        # have associated item data, to avoid memory leaks.
        #
        # id: item ID or a tag
        id_set = self.__tag_idset_map.pop(id, None)
        if id_set:
            # id is a tag, id_set is a set of item IDs for that tag
            for item_id in id_set:
                self.__id_tagtuple_map.pop(item_id, None)
                self.__id_data_map.pop(item_id, None)
        else:
            data = self.__id_data_map.pop(id, None)
            if data:
                unbind = getattr(data, 'unbind', None)
                if unbind:
                    # Remove event binding to text entry
                    unbind('<KeyPress>')
                # id is the original item ID
                for tag in self.__id_tagtuple_map.pop(id, []):
                    self.__tag_idset_map.pop(tag, {}).pop(id, None)

    def __resetItemData(self):
        self.__id_data_map = {} # map item ID to extra data
        self.__id_tagtuple_map = {} # map item ID to its tags
        self.__tag_idset_map = {} # map tag to item ID set

    def __storeItemData(self, id, data, tag_list):
        # Store an object associated with a DrawingWindow item in such a way
        # that we can clean it up whether it is deleted by ID, tag, or
        # containing object's tag. This is needed i.e. to keep the
        # PhotoImage object around so Image items don't disappear from the
        # Canvas.
        #
        # id: the original ID of the DrawingWindow item
        # data: the data object
        # tag_list: the list of currently active tags
        tag_tuple = tuple(tag_list)
        self.__id_data_map[id] = data
        self.__id_tagtuple_map[id] = tag_tuple
        for tag in tag_tuple:
            self.__tag_idset_map.setdefault(tag, {})[id] = True

    def _createFont(self, point_size):
        fontfamily = 'Courier'
        fontweight = 'normal'
        return Font(root=self.window,
                font=(fontfamily, point_size, fontweight))

    def _createSizeFontMap(self):
        # Return { size: font }
        result = {}
        fontsizes = (self.HUGE, self.LARGE, self.MEDIUM, self.SMALL)
        for font_size in fontsizes:
            point_size = font_size # Implementation detail!
            font = self._createFont(point_size)
            result[font_size] = font
        return result

    def _clear(self, force=False):
        self.__resetItemData()
        if self.__tag_stack and self.canvas:
            self.canvas.delete(self.__tag_stack[0])
        self.__tag_stack = ['__ALL__']
        if force and self.canvas:
            # delete any items not marked with known tags
            canvas_items = self.canvas.find_all()
            #if canvas_items:
            #    print "DrawingWindow._clear: deleting leftover items",
            for item in canvas_items:
            #    sys.stdout.write('.')
                self.canvas.delete(item)

    def _doClose(self, event=None):
        self._clear()
        self.__size_font_map = {}
        if self.window:
            # XXX sometimes segfaults on ActivePython on Linux
            self.window.destroy()
        self.window = self.canvas = None
    
    def _onConfigure(self, event=None):
        if event and event.widget is self.canvas:
            self.width = event.width
            self.height = event.height
            if self.__first_configure:
                if self.__startup_callback:
                    self.__startup_callback()
                self.__first_configure = False
            if self.__resize_callback:
                self.__resize_callback()

    def _onDeleteWindow(self, event=None):
        if self.__delete_callback:
            callback = self.__delete_callback
            self.__delete_callback = None # make sure it's not called twice
            callback()
        self._doClose()

    def _setTitle(self, text):
        self.__title = text
        if self.window:
            self.window.title(self.__title)
            # Make the drawing commands take effect
            if self.interactive:
                self.window.update()

    def _getFont(self, font_param):
        if isinstance(font_param, Font):
            return font_param
        font = self.__size_font_map.get(font_param)
        if font:
            return font
        if isinstance(font_param, (int, float)):
            font = self._createFont(font_param)
            self.__size_font_map[font_param] = font
            return font
        raise ValueError("Bad font parameter: %s" % repr(font_param))

    def afterDelayCall(self, delay_sec, callback, *params, **kwparams):
        """
        Start a timer that will cause a function to be called after an
        approximate amount of time has passed.
        
        delay_sec:
            The time delay until the function will be called, in
            seconds.
        callback:
            The function to call after the delay has passed. Give the
            name of the function without parenthesis.
        *params, **kwparams:
            (Optional) The parameters that will be passed to the callback
            function, if any.

        Return a Timer object. You can call the stop() method on the timer
        object before the delay passes if you want to stop timer from calling
        the function.

        Example:
            >>> # print a message after a 2 second delay
            >>> # close the window after 2 more seconds
            >>> from cpif.graphics import DrawingWindow
            >>> dw = DrawingWindow(100, 100)
            >>> def func(data):
            ...    dw = data
            ...    dw.drawText(0, 0, "Timer went off after 2 seconds")
            ... 
            >>> timer = dw.afterDelayCall(2.0, func, dw)
            >>> dw.runtest(delayseconds=3.0)
            >>> dw.close()
        """
        try:
            delay_ms = int(delay_sec * 1000.)
        except ValueError:
            raise ValueError(
                "The delay_sec parameter must be a valid number of "
                "seconds, not %s." % repr(delay_sec))
        if delay_sec < 0:
            raise ValueError(
                "The delay_sec parameter must not be negative.")
        _checkParams(callback, len(params), kwparams)
        def _callback():
            return callback(*params, **kwparams)
        timer = Timer(self, delay_sec, _callback, one_shot=True)
        timer.start()
        return timer

    def arc(self, x, y, radius, start, extent, color,
            fill='', width=1, style=PIESLICE):
        """
        Draw an arc (a part of a circle).
        Return an ID usable by delete().
        
        x, y: the coordinates of the center of the arc
        
        radius: the distance in pixels from the center to the edge of arc
        
        start: The beginning of the angular range of the arc, measured in
               degrees counter-clockwise from the three-o'-clock position.
               It may be either positive or negative.

        extent: The size of the angular range of the arc. The arc extends
                (extent) degrees counter-clockwise from the (start) angle.

        color: The color to draw the rim of the arc

        fill: The color to fill inside the arc, default is transparent 
              (no fill)

        width: The width in pixels of the rim of the arc (default is 1)

        style: The manner in which the arc is drawn:
                 DrawingWindow.PIESLICE (default):
                   Draw around the rim of the circle from the start point to
                   the end of the extent, then two line segments between the
                   center and the start and end positions.
                 DrawingWindow.CHORD:
                   Draw around the rim of the circle from the start point to
                   the end of the extent, then draw one line segment between
                   the start and end positions.
                 DrawingWindow.ARC:
                   Just draw around the rim of the circle from the start
                   point to the end of the extent. Do not fill, fill color
                   is ignored.

        Example:
            >>> from cpif.graphics import DrawingWindow
            >>> dw = DrawingWindow(200, 200)
            >>> id = dw.arc(100, 100, 50, 45, 270, 'yellow', fill='yellow')
            >>> dw.runtest(delayseconds=1.0)
            >>> dw.close()

        Draws a Pac-Man(tm)-like figure with mouth facing right.
        """
        color = _convertColor(color)
        fill = _convertColor(fill)
        x1 = x - radius
        y1 = y - radius
        x2 = x + radius
        y2 = y + radius
        id = self.canvas.create_arc(x1, y1, x2, y2, extent=extent,
                                      outline=color, start=start,
                                      style=style, fill=fill, width=width,
                                      tag=tuple(self.__tag_stack))
        # Make the drawing commands take effect
        if self.interactive:
            self.window.update()
        return id

    def bbox(self, id):
        """
        Return the "bounding box" - a tuple of (X1,Y1,X2,Y2) coordinates for
        a rectangle which encloses the drawing with the given id.
        """
        return self.canvas.bbox(id)

    def beginDrawing(self, id=None):
        """
        Begin creating a new drawing, which can be composed of multiple
        lines, boxes, circles, etc. If id is not given, a new ID is
        automatically generated. (Supplying an ID is useful if you want to
        add to an existing drawing for which you have the ID.)

        Return the ID of the drawing, which can be used by centerOf(),
        moveTo(), moveBy(), delete(), etc.

        Every drawing method that is called from when you call
        beginDrawing() to when you call endDrawing() creates an object that
        is part of the drawing. Each part of the drawing is affected by any
        operation using the ID returned by beginDrawing(). You can nest one
        drawing inside of another, as long as you have an endDrawing() call
        for every beginDrawing().

        Example:
            from cpif.graphics import DrawingWindow
            dw = DrawingWindow(100, 100, interactive=True)
            id = dw.beginDrawing()
            # both box and circle are part of the same drawing
            dw.box(10, 10, 90, 90, 'black')
            dw.circle(40, 40, 20, 'red')
            dw.endDrawing()
            # shift box and circle 10 pixels right and 5 down
            dw.moveBy(id, 10, 5)
            # delete box and circle
            dw.delete(id)
        """
        if id is None:
            # auto-generate a tag name
            self.__tag_seq_num += 1
            id = 'tag-%d' % self.__tag_seq_num
        self.__tag_stack.append(id)
        return self.__tag_stack[-1]

    def box(self, x1, y1, x2, y2, color, fill='', width=1):
        """
        Draw a box (rectangle) on the canvas.
        Return the ID of the rectangle, which can be used by delete().
        x1, y1: the coordinates of the upper-left corner
        x2, y2: the coordinates of the lower-right corner
        color: the color of the rectangle outline
        fill: the fill color of the rectangle, defaults to transparent
        width: the width of the rectangle outline, in pixels, defaults to 1
        """
        color = _convertColor(color)
        fill = _convertColor(fill)
        id = self.canvas.create_rectangle(x1, y1, x2, y2, outline=color,
                                            fill=fill, width=width,
                                            tag=tuple(self.__tag_stack))
        # Make the drawing commands take effect
        if self.interactive:
            self.window.update()
        return id

    def centerOf(self, id):
        """
        Return the coordinates of the center of a drawing. The center is
        calculated by taking the average of the coordinates of all of the
        components of the drawing.

        id: an ID number returned by a drawing method, 
            or an ID returned by beginDrawing()

        ValueError is raised if id is not a valid drawing ID.
        """
        tx, ty = 0, 0
        coords = self.canvas.coords(id)
        n = len(coords) // 2
        if n == 0:
            raise ValueError('%s is not a valid drawing ID' % repr(id))
        for i in xrange(0, len(coords), 2):
            tx += coords[i]
            ty += coords[i + 1]
        return (tx / n, ty / n)

    def circle(self, x, y, radius, color, fill='', width=1):
        """Draw a circle on the canvas.
        Return the ID of the circle, which can be used by delete().
        x, y: the center coordinates of the circle
        radius: the radius of the circle
        color: the color of the outline of the circle. If color is an
            empty string then no outline will be drawn for the circle.
        fill: the fill color of the circle, default is to not fill in
            the circle.
        width: the width of the circle outline, in pixels, defaults to 1
        """
        color = _convertColor(color)
        fill = _convertColor(fill)
        id = self.canvas.create_oval(x - radius, y - radius,
                                       x + radius, y + radius,
                                       outline=color, fill=fill,
                                       width=width,
                                       tag=tuple(self.__tag_stack))
        # Make the drawing commands take effect
        if self.interactive:
            self.window.update()
        return id

    def clear(self):
        """
        Erase the contents of the drawing window.

        >>> from cpif.graphics import DrawingWindow
        >>> dw = DrawingWindow(100, 100, interactive=True)
        >>> id = dw.circle(50, 50, 50, 'red', fill='red')
        >>> for radius in range(0, 50, 2):
        ...     id = dw.circle(50, 50, radius, 'black')
        ...
        >>> # See the pretty picture for two seconds
        >>> import time; time.sleep(2.0)
        >>> # Now it will disappear
        >>> dw.clear()
        >>> time.sleep(2.0)
        >>> dw.close()
        >>> time.sleep(0.5)
        """
        # Delegate to an internal method -- this is done so that _doClose
        # only calls methods whose names begins with _, so that the turtle
        # module _killInstance() function works properly.
        self._clear()
        if self.interactive and self.window:
            self.window.update()

    def close(self):
        """Close the graphics window, stop processing events."""
        self._doClose(None)

    def delete(self, id):
        """
        Delete an item that had been drawn on the canvas.

        id: an ID number returned by a drawing method, 
            or an ID returned by beginDrawing()
        """
        self.__cleanupItemData(id)
        if self.canvas:
            self.canvas.delete(id)
        if self.interactive and self.window:
            self.window.update()

    def drawImage(self, x, y, imagefile):
        """
        Draw an image on the canvas.

        x, y: the pixel coordinates of the upper left corner of the image
        imagefile: the name of the .gif or .bmp image file to display

        If imagefile does not include the full directory path of the file,
        then the current directory, the program directory, and the
        cpif.images() directory are searched for the name of the image file.
        
        Returns the ID of the image.

        example:
            >>> # Display the cpif logo in a DrawingWindow
            >>> from cpif.graphics import DrawingWindow
            >>> dw = DrawingWindow(350, 350)
            >>> image_id = dw.drawImage(20, 25, 'logo1.gif')
            >>> dw.runtest(delayseconds=1.0)
            >>> dw.close()
        """
        if isinstance(imagefile, basestring):
            ext = os.path.splitext(imagefile)[1]
            if not ext.lower() in SUPPORTED_IMAGE_FILE_EXTENSIONS:
                raise ValueError(
                        "%r is not a supported image file extension" % ext)
            fullfile = findImageFile(imagefile)
            if not fullfile:
                raise IOError("File not found: %s" % imagefile)
            image = Tk.PhotoImage(file=fullfile, master=self.canvas)
        else:
            image = imagefile # could be PIL.ImageTk.PhotoImage object
        id = self.canvas.create_image(x, y, anchor=Tk.NW,
                                      image=image,
                                      tag=tuple(self.__tag_stack))
        # We put the image in a map in order to keep it from being
        # garbage-collected, otherwise there is a blank spot on the
        # canvas where the image should be displayed (honest, I tried it!)
        self.__storeItemData(id, image, self.__tag_stack)
        # Make the drawing commands take effect
        if self.interactive:
            self.window.update()
        return id

    def drawText(self, x, y, text, color='black', font_size=MEDIUM):
        """
        Draw text on the canvas.

        x, y: the pixel coordinates where the text will be drawn
        text: the string of text to draw
        color: (optional) the color in which the text will be drawn, the
               default is 'black'.
        font_size: (optional) the font size, one of:
            DrawingWindow.HUGE, LARGE, MEDIUM, SMALL. The default is
            DrawingWindow.MEDIUM.
        Returns the ID of the text item

        example:
            >>> import cpif.graphics
            >>> dw = cpif.graphics.DrawingWindow(100, 100)
            >>> id = dw.drawText(0, 0, 'hello', 'green')
            >>> dw.runtest(delayseconds=1.0)
            >>> dw.close()

        Draws the word "hello" in green in the upper-left corner of a
        100x100 pixel window.
        """
        color = _convertColor(color)
        id = self.canvas.create_text(x, y, text=text, anchor=Tk.NW,
                                       fill=color,
                                       font=self._getFont(font_size),
                                       tag=tuple(self.__tag_stack))
        # Make the drawing commands take effect
        if self.interactive:
            self.window.update()
        return id

    def endDrawing(self):
        """
        Stop adding to a drawing. Return the id of the previous drawing in
        which this drawing was nested, or None if this was a top-level
        drawing.
        
        See beginDrawing() for more information.
        """
        if self.__tag_stack:
            del self.__tag_stack[-1]
        if self.__tag_stack:
            return self.__tag_stack[-1]
        else:
            return None

    def getCoords(self, id):
        """
        Return the coordinates of an object as a list of coordinate pairs.
        Usually the result will be in the form:
            [(x1, y1), (x2, y2)]
        where (x1, y1) is the upper-left "corner" of the object, or starting
        point of the line, and (x2, y2) is the lower left "corner" of the
        object, or end of the line. Different objects have different numbers
        of coordinate points; try this method on the various object types to
        find out exactly how many.

        example:
            >>> from cpif.graphics import DrawingWindow
            >>> dw = DrawingWindow(100, 100, interactive=True)
            >>> id = dw.oval(10, 20, 90, 80, 'black', fill='violet')
            >>> coords = dw.getCoords(id)
            >>> for (x, y) in coords:
            ...     print "%.1f, %.1f" % (x, y)
            ... 
            10.0, 20.0
            90.0, 80.0
            >>> dw.close()
        """
        pl = self.canvas.coords(id)
        return [(pl[i], pl[i+1]) for i in xrange(0, len(pl), 2)]

    def getImageSize(self, image_id_or_file):
        """
        Return the size of an image as a tuple containing (width, height) in
        pixels.

        image_id_or_file:
            The ID of the image returned by the drawImage() method, or the
            name of an image file (as used by drawImage()).

        >>> from cpif.graphics import DrawingWindow
        >>> dw = DrawingWindow(200, 200)
        >>> dw.getImageSize('PythonPowered.gif')
        (110, 44)
        >>> dw.close()
        """
        if isinstance(image_id_or_file, basestring):
            fullfile = findImageFile(image_id_or_file)
            if not fullfile:
                raise IOError("File not found: %s" % image_id_or_file)
            image = Tk.PhotoImage(file=fullfile)
        else:
            image = self.__id_data_map.get(image_id_or_file, None)
            if not image:
                raise KeyError("%s is not a valid image ID" % 
                               repr(image_id_or_file))
        return (image.width(), image.height())

    def getSize(self):
        """
        Return the size of the DrawingWindow drawing canvas, as a
        tuple containing (width, height) in pixels.

        example:
            >>> from cpif.graphics import DrawingWindow
            >>> dw = DrawingWindow(173, 69, interactive=True)
            >>> width, height = dw.getSize()
            >>> print width, height
            173 69
            >>> dw.close()
        """
        width = self.canvas.winfo_width()
        height = self.canvas.winfo_height()
        return (width, height)

    def getScreenSize(self):
        """
        Return the size of the screen in which the DrawingWindow is
        displayed, as a tuple containing (width, height) in pixels.
        """
        width = self.canvas.winfo_screenwidth()
        height = self.canvas.winfo_screenheight()
        return (width, height)

    def getText(self, id):
        """
        Return the text string associated with an item created with
        drawText() or textEntry().
        id: The ID of the text or text entry object
        """
        itemdata = self.__id_data_map.get(id)
        cget = getattr(itemdata, 'cget', None)
        if cget:
            try:
                textvariable = cget('textvariable')
            except Tk.TclError:
                # no textvariable
                pass
            else:
                return self.canvas.getvar(textvariable)
        # id was created by drawText()
        return self.canvas.itemcget(id, 'text')

    def getTextEntrySize(self, text, font_size=MEDIUM):
        """Estimate the size that would be required by a text entry window,
        given a text string. Return (width, height) in pixels.
        """
        w, h = self.getTextSize(text, font_size=font_size)
        return (w + int(2*self.TEXTENTRY_PADX), h + int(2*self.TEXTENTRY_PADY))

    def getTextSize(self, text, font_size=MEDIUM):
        """
        Return the size in pixels that a text string would occupy on the
        canvas if it were to be drawn using drawText().

        text: the string
        font_size: (optional) the font size, one of:
            DrawingWindow.HUGE, LARGE, MEDIUM, SMALL. The default is
            DrawingWindow.MEDIUM.

        Returns (width, height) in pixels.

        Example:
            >>> from cpif.graphics import DrawingWindow
            >>> dw = DrawingWindow(100, 100, interactive=True)
            >>> text = "Hello!"
            >>> id = dw.drawText(0, 0, text)
            >>> width, height = dw.getTextSize(text)
            >>> # display the width and height under the text
            >>> id2 = dw.drawText(0, height, repr((width, height)))
            >>> import time; time.sleep(1.0)
            >>> dw.close()
        """
        font = self._getFont(font_size)
        width = font.measure(text)
        height = font.metrics('linespace')
        return (width, height)

    def hasFocus(self, id):
        """
        Return True if id identifies a text entry and it currently has the
        keyboard focus. Return True if id is None and setFocus(None) was
        called. Raise ValueError if id is not None and is not a valid text
        entry ID. Otherwise return False.

        See also textEntry() and setFocus().
        """
        if id is None:
            return self.canvas.focus_get() == self.canvas
        entry = self.__id_data_map.get(id)
        if not entry:
            raise ValueError("bad id: not a valid drawing window item")
        if not hasattr(entry, 'focus_get'):
            raise ValueError("bad id: not a text entry")
        return entry.focus_get() == entry

    def line(self, x1, y1, x2, y2, color, width=1):
        """
        Draw a line on the canvas. Return the ID of the line, which
        can be used later by the delete() method.

        x1, y1: the coordinates of the starting point of the line
        x2, y2: the coordinates of the ending point of the line
        color: A color, such as 'red', (255, 128, 37), etc.

        example:
            >>> # Draw a line from the upper left to lower right
            >>> from cpif.graphics import DrawingWindow
            >>> dw = DrawingWindow(100, 100)
            >>> dw.title('test')
            >>> id = dw.line(20, 30, 80, 90, 'black')
            >>> dw.runtest(delayseconds=1.0)
            >>> dw.close()
        """
        color = _convertColor(color)
        kwparams = {'fill': color, 'width': width,
                    'tag': tuple(self.__tag_stack)}
        id = self.canvas.create_line(x1, y1, x2, y2, **kwparams)
        # Make the drawing commands take effect
        if self.interactive:
            self.window.update()
        return id

    def lines(self, point_list, color, width=1):
        """
        Draw a line on the canvas. Return the ID of the line, which
        can be used later by the delete() method.

        point_list:
            A sequence of (x,y) tuples, one for each endpoint or corner
            of the line. There must be at list two tuples in the list.
        color: A color, such as 'red', (255, 128, 37), etc.

        example:
            >>> # Draw a line from the upper left to lower right
            >>> from cpif.graphics import DrawingWindow
            >>> dw = DrawingWindow(100, 100)
            >>> dw.title('test')
            >>> point_list = [(25, 75), (75, 25), (25, 25), (75, 75)]
            >>> id = dw.lines(point_list, 'black')
            >>> dw.runtest(delayseconds=1.0)
            >>> dw.close()
        """
        color = _convertColor(color)
        coords = [coord for coord_pair in point_list
                          for coord in coord_pair]
        kwparams = {'fill': color, 'width': width,
                    'tag': tuple(self.__tag_stack)}
        id = self.canvas.create_line(*coords, **kwparams)
        # Make the drawing commands take effect
        if self.interactive:
            self.window.update()
        return id

    def moveTo(self, id, x, y):
        """
        Move the center of a drawing to a new location.

        id: an ID number returned by a drawing method, 
            or an ID returned by beginDrawing().
        x, y: The new center location of the drawing.

        ValueError is raised if id is not a valid drawing ID.
        """
        cx, cy = self.centerOf(id)
        self.moveBy(id, x - cx, y - cy)

    def moveBy(self, id, deltaX, deltaY):
        """
        Move a drawing deltaX pixels horizontally (positive or negative)
        and deltaY pixels vertically (positive or negative).

        id: an ID number returned by a drawing method, 
            or an ID returned by beginDrawing().
        deltaX, deltaY: The horizontal and vertical distance, in pixels,
            that the drawing will be moved.
        """
        self.canvas.move(id, deltaX, deltaY)

    def onClickCall(self, id, callback, *params, **kwparams):
        """
        Set a function to be called when a drawing is clicked.

        id: an ID number returned by a drawing method, 
            or an ID returned by beginDrawing().

        callback:
            The function to call when the left mouse button is pressed over 
            the drawing. It will be called like this:

                callback(mouse_event)

            mouse_event is an object with the following attributes:
                button_num:
                    1 for left button, 2 for middle, 3 for right,
                    0 for no button
                x: the X window coordinate of the mouse event
                y: the Y window coordinate of the mouse event

            If callback is None then any function previously associated
            with this event is deactivated.

        *params, **kwparams:
            (Optional) The parameters that will be passed to the callback
            function after the mouse_event parameter.
        """
        if callback:
            _checkParams(callback, len(params) + 1, kwparams)
            def _callback(event):
                return callback(MouseEvent(event), *params, **kwparams)
            self.canvas.tag_bind(id, '<1>', _callback)
        else:
            self.canvas.tag_unbind(id, '<1>')

    def onDeleteWindowCall(self, callback):
        """
        Set a function or method to be called when the window is deleted by
        the user (i.e. by clicking the x button in the upper-left corner.)

        The callback function is *not* called when the DrawingWindow close()
        method is called by the program.  Also, close() is called for you
        automatically after your callback function returns, so you do not
        need to call close().  If callback is None then no function will be
        called.

        The callback function does not take any parameters.
        """
        if callback:
            _checkParams(callback, 0)
        self.__delete_callback = callback

    def onEvent(self, *params):
        """Event-tracing method."""
        print "onEvent", params
        def _dumpObj(name, obj, indent='', objset=None):
            print indent + name, repr(obj)
            if isinstance(obj, (int, long, float, basestring, 
                    list, tuple, dict)):
                # Don't recurse on "simple" built-in objects
                return
            if objset is None:
                objset = {}
            for name in dir(obj):
                if (name.startswith('_') or 
                    name.startswith('func_') or
                    name.startswith('im_')):
                    continue
                attr = getattr(obj, name)
                if callable(attr):
                    # exclude methods
                    continue
                if id(attr) in objset:
                    continue
                objset[id(attr)] = True
                _dumpObj(name, attr, indent + '  ', objset)
        def _dumpEvent(event):
            for name in dir(event):
                if name.startswith('_'):
                    continue
                # hide the Tkinter event attribute from novices
                if name == 'event':
                    continue
                print '  ', name, repr(getattr(event, name))
        i = 0
        for param in params:
            if isinstance(param, Event):
                print "Event param %d: Event object" % i
                _dumpEvent(param)
            else:
                _dumpObj("Event param %d:" % i, param)
            i += 1

    def onKeyDownCall(self, callback, *params, **kwparams):
        """
        Set a function to call when a key is pressed on the keyboard.

        callback:
            The function to call when a key is pressed. It will be called
            like this:

                callback(key_event)

            key_event is an object with the following attributes:
                char: A string containing the character for this key
                keysym: A string describing this key, i.e. 'Escape'.

            If callback(key_event) returns a True value, then no other
            window will get this key event. If the callback returns a False
            value (or None, etc) then other windows in your program may also
            get this key event. If in doubt, don't return any value (same as
            returning None).

            If the callback parameter is None then any function previously
            associated with this event is deactivated.

        *params, **kwparams:
            (Optional) Extra parameters to pass to the callback function
            after the key_event parameter.
        """
        if callback:
            _checkParams(callback, len(params) + 1, kwparams)
            def _callback(event):
                if callback(KeyEvent(event), *params, **kwparams):
                    return 'break'
                else:
                    return ''
            self.window.bind('<KeyPress>', _callback)
        else:
            self.window.unbind('<KeyPress>')

    def onKeyUpCall(self, callback, *params, **kwparams):
        """
        Set a function to call when a key is released on the keyboard.

        callback:
            The function to call when a key is released. It will be called
            like this:

                callback(key_event)

            See onKeyDownCall() for more details on callback and key_event.

            If the callback parameter is None then any function previously
            associated with this event is deactivated.

        *params, **kwparams:
            (Optional) The parameters that will be passed to the callback
            function after the key_event parameter.
        """
        if callback:
            _checkParams(callback, len(params) + 1, kwparams)
            def _callback(event):
                if callback(KeyEvent(event), *params, **kwparams):
                    return 'break'
                else:
                    return ''
            self.window.bind('<KeyRelease>', _callback)
        else:
            self.window.unbind('<KeyRelease>')

    def onMouseDownCall(self, callback, *params, **kwparams):
        """
        Set a function to call when a button is pressed on the mouse.

        callback:
            The function to call when a mouse button is pressed. It will
            be called like this:

                callback(mouse_event)

            mouse_event is an object with the following attributes:
                button_num: 1 for left button, 2 for middle, 3 for right
                x: the X window coordinate of the mouse event
                y: the Y window coordinate of the mouse event

            If callback is None then any function previously associated
            with this event is deactivated.

        *params, **kwparams:
            (Optional) The parameters that will be passed to the callback
            function after the mouse_event parameter.
        """
        if callback:
            _checkParams(callback, len(params) + 1, kwparams)
            def _callback(event):
                return callback(MouseEvent(event), *params, **kwparams)
            self.canvas.bind('<ButtonPress>', _callback)
        else:
            self.canvas.unbind('<ButtonPress>')

    def onMouseUpCall(self, callback, *params, **kwparams):
        """
        Set a function to call when a button is released on the mouse.

        callback:
            The function to call when a mouse button is released. It will
            be called like this:

                callback(mouse_event)

            See onMouseDownCall() for more details on mouse_event.

            If callback is None then any function previously associated
            with this event is deactivated.

        *params, **kwparams:
            (Optional) The parameters that will be passed to the callback
            function after the mouse_event parameter.
        """
        if callback:
            _checkParams(callback, len(params) + 1, kwparams)
            def _callback(event):
                return callback(MouseEvent(event), *params, **kwparams)
            self.canvas.bind('<ButtonRelease>', _callback)
        else:
            self.canvas.unbind('<ButtonRelease>')

    def onMouseMoveCall(self, callback, *params, **kwparams):
        """
        Set a function to call when the mouse is moved.
        This will get called a *lot*, beware!

        callback:
            The function to call whenever the mouse is moved. It will
            be called like this:

                callback(mouse_event)

            See onMouseButtonDownCall() for more details on mouse_event.
            mouse_event.button_num will be 0.

            If callback is None then any function previously associated
            with this event is deactivated.

        *params, **kwparams:
            (Optional) The parameters that will be passed to the callback
            function after the mouse_event parameter.
        """
        if callback:
            _checkParams(callback, len(params) + 1, kwparams)
            def _callback(event):
                return callback(MouseEvent(event), *params, **kwparams)
            self.canvas.bind('<Motion>', _callback)
        else:
            self.canvas.unbind('<Motion>')

    def onResizeCall(self, callback):
        """
        Set a function or method that will be called when the window changes
        size, and when it is displayed for the first time.  If callback is
        None then no function will be called.

        The callback function does not take any parameters.

        The callback function can get the new width and height of this
        DrawingWindow by calling dw.getSize() or by using dw.width and
        dw.height (where dw is the name of your DrawingWindow).
        """
        if callback:
            _checkParams(callback, 0)
        self.__resize_callback = callback

    def onTimerCall(self, interval_sec, callback, *params, **kwparams):
        """
        Set a timer that will call a function at regular intervals.

        interval_sec: The time interval of the timer, in seconds.
        callback: The function that the timer will call.
        *params, **kwparams:
            (Optional) The parameters that will be passed to the callback
            function, if any.

        Return a timer object. You can stop the timer by calling the stop()
        method on the timer object.

        Example:
        >>> from cpif.graphics import DrawingWindow
        >>> dw = DrawingWindow(100, 100)
        >>> def callback(dw):
        ...     dw.drawText(0, 0, "timer function")
        ...     timer.stop()
        ... 
        >>> timer = dw.onTimerCall(1.0, callback, dw)
        >>> # Wait for message to be displayed
        >>> dw.runtest(delayseconds=1.5)
        >>> dw.close()
        """
        try:
            interval_ms = int(interval_sec * 1000.)
        except ValueError:
            raise ValueError(
                "The interval_sec parameter must be a valid number of "
                "seconds.")
        if interval_sec < 0:
            raise ValueError(
                "The interval_sec parameter must not be negative.")
        _checkParams(callback, len(params), kwparams)
        def _callback():
            return callback(*params, **kwparams)
        timer = Timer(self, interval_sec, _callback)
        timer.start()
        return timer

    def oval(self, x1, y1, x2, y2, color, fill='', width=1):
        """
        Draw an oval on the canvas.
        Return the ID of the oval, which can be used by delete().

        x1, y1: The coordinates of the upper left "corner" of the oval
        x2, y2: The coordinates of the lower right "corner" of the oval
        color: the color of the outline of the oval. If color is an
            empty string then no outline will be drawn for the oval.
        fill: the fill color of the oval, default is to not fill in
            the oval.
        width: the width of the oval outline, in pixels, defaults to 1
        """
        color = _convertColor(color)
        fill = _convertColor(fill)
        id = self.canvas.create_oval(x1, y1, x2, y2,
                                       outline=color, fill=fill,
                                       width=width,
                                       tag=tuple(self.__tag_stack))
        # Make the drawing commands take effect
        if self.interactive:
            self.window.update()
        return id

    def polygon(self, point_list, color, fill='', width=1):
        """
        Draw a polygon.
        Return the ID of the polygon, which can be used by delete().

        point_list:
            A sequence of (x,y) tuples, one for each vertex (corner)
            of the polygon.
        color:
            The color of the outline of the polygon. If color is an
            empty string then no outline will be drawn for the polygon.
        fill:
            The fill color of the polygon, default is to not fill in
            the polygon.
        width: the width of the circle outline, in pixels, defaults to 1

        example:
            >>> from cpif.graphics import DrawingWindow
            >>> dw = DrawingWindow(200, 200)
            >>> # draw a hollow red triangle
            >>> id = dw.polygon([(100, 50), (150, 150), (50, 150)], 'red')
            >>> # draw a solid blue diamond
            >>> id = dw.polygon([(100, 50), (150, 100), (100, 150), (50, 100)],
            ...            'blue', fill='blue')
            ... 
            >>> dw.runtest(delayseconds=1.0)
            >>> dw.close()
        """
        kwparams = {'outline': color, 'fill': fill, 'width': width,
                    'tag': tuple(self.__tag_stack)}
        id = self.canvas.create_polygon(*point_list, **kwparams)
        # Make the drawing commands take effect
        if self.interactive:
            self.window.update()
        return id

    def resize(self, width, height):
        """
        Change the size of the DrawingWindow.

        width: The new width of the window, in pixels.
        height: The new height of the window, in pixels.
        """
        self.canvas.configure(width=width, height=height)
        self.window.configure(width=width, height=height)
        # Make the drawing commands take effect
        if self.interactive:
            self.window.update()

    def run(self):
        """Run the graphics program.
        Return when the window has been closed.
        """
        self.window.mainloop()

    def runInBackground(self):
        """Start the graphics window but keep running the main program as well.
        This is useful when testing from the Python command prompt.

        This method only exists for backwards compatibility with older
        versions of the book. Pass the parameter interactive=True to
        DrawingWindow to achieve the same effect in newer code.
        """
        self.interactive = True
        self.window.update()

    # A shorter name for runInBackground() (which is now obsolete)
    runbg = runInBackground

    def runtest(self, delayseconds=2.0):
        """Start the graphics program, but close it again after
        a few seconds, as determined by the delayseconds parameter.

        This is useful for automated testing.
        """
        self.window.after(int(delayseconds * 1000), self.close)
        self.run()

    def saveFrame(self, base_filename='frame-', frame_num=None,
            num_digits=4, filename_end='.eps', verbose=False):
        """
        Save the current DrawingWindow contents to a graphics file in
        Encapsulated PostScript format. 

        The filename is base_filename+frame_num+filename_end

        Where:
            base_filename --
                The first part of the filename to which the frame will be
                saved, defaults to 'frame-' (in the current directory). If
                None is passed, then the data is not saved to a file but
                returned from this method as a string. Defaults to 'frame-'.
            frame_num  --
                A frame number to append to the base filename.  Defaults to the
                last frame number + 1. If a string is given, the string is used
                in place of a frame number, and the last frame number is not
                changed.
            num_digits -- 
                The number of digits to pad the frame number to, default is 4.
                If a string is passed as the frame_num parameter, then this
                parameter has no effect.
            filename_end --
                The last part of the name of the file to which the frame will
                be saved. Defaults to ".eps" (for Encapsulated PostScript)
            verbose -- 
                If true, print the filename before saving, defaults to False

        Example:
            If you create a DrawingWindow and then call this method three
            times without any parameters (in between other drawing commands,
            of course), the graphics will be saved to the following three
            files:
                frame-0000.eps
                frame-0001.eps
                frame-0002.eps
            etc.
        """
        if self.interactive:
            self.window.update()
        w, h = self.getSize()
        kwparams = {'x': 0, 'y': 0, 'width': w, 'height': h,
                    'pagex': 0, 'pagey': 0, 'pagewidth': w, 'pageheight': h,
                    'pageanchor': 'nw'}
        if base_filename is not None:
            if frame_num is None:
                frame_num = self.__last_frame_num + 1
            if isinstance(frame_num, basestring):
                frame_num_str = frame_num
            else:
                frame_num_fmt = "%%0%dd" % num_digits
                frame_num_str = frame_num_fmt % frame_num
                self.__last_frame_num = frame_num
            filename = "%s%s%s" % (base_filename, frame_num_str, filename_end)
            if filename:
                kwparams['file'] = filename
                if verbose:
                    print filename
        return self.canvas.postscript(**kwparams)

    def setCoords(self, id, coords):
        """
        Set the coordinates of an object. This can change both the size
        and position of an object.

        id: an ID number returned by a drawing method, 
            or an ID returned by beginDrawing()
        coords:
            A sequence of coordinate pairs giving the new coordinates of the
            object. Call getCoords() to find out the exact number of
            coordinates an object type has.

        Note:
            A circle is really an oval, so set its coordinates the same way.

        Example:
            >>> # Draw a box, then move it 20 pixels down and left
            >>> from cpif.graphics import DrawingWindow
            >>> x1, y1, x2, y2 = (10, 15, 60, 55)
            >>> dw = DrawingWindow(100, 100)
            >>> box = dw.box(x1, y1, x2, y2, 'black')
            >>> def moveBox(dw):
            ...     dw.setCoords(box, [(x1+20, y1+20), (x2+20, y2+20)])
            ...
            >>> timer = dw.afterDelayCall(1.0, moveBox, dw)
            >>> dw.runtest(delayseconds=2.0)
            >>> dw.close()
        """
        coord_list = [coord for coord_pair in coords
                                for coord in coord_pair]
        self.canvas.coords(id, *coord_list)
        if self.interactive:
            self.window.update()

    def setFocus(self, id):
        """
        If id identifies a text entry, set the keyboard focus to that item.
        If id is None, set the focus to the drawing window itself (thus
        unsetting focus from any text entry in the drawing window.)
        """
        if id is None:
            self.canvas.focus_set()
        focus_set = getattr(self.__id_data_map.get(id), 'focus_set', None)
        if focus_set:
            focus_set()

    def setText(self, id, text):
        """Set or change the text associated with an item created with
        drawText() or textEntry().
        id: The ID of the text or text entry object.
        text: The new text for the item.
        """
        itemdata = self.__id_data_map.get(id)
        cget = getattr(itemdata, 'cget', None)
        if cget:
            try:
                textvariable = cget('textvariable')
            except Tk.TclError:
                pass
            else:
                self.canvas.setvar(textvariable, text)
                return
        # This is an item created by drawText()
        self.canvas.itemconfigure(id, text=text)
        if self.interactive:
            self.window.update()

    def title(self, text):
        """Set the title text of the drawing window."""
        if self.window:
            self.window.after(0, self._setTitle, text)
        else:
            # the title will be set later when the window is created
            self.__title = text

    def textEntry(self, x, y, width=None, size=None, set_focus=False, 
                  prompt='', keypress_callback=None, font_size=MEDIUM):
        """
        Create a 1-line text entry box. Return its ID.
        x, y: Coordinates of upper-left corner of text entry box
        width: Width, in characters, of the text entry box
        size: (width, height) size in pixels of the text entry box
        Either width or size must be specified, but not both.
        set_focus: (optional) if true, set the keyboard input focus to this
                   text entry box. Default is False.
        prompt: (optional) label to display to left of text entry box
        keypress_callback: (optional) function to call when a key is pressed
            when the text entry box has the keyboard focus. The function is
            called with exactly one parameter, the key_event parameter. See
            onKeyDownCall() for more details.
        font_size: (optional) the font size, one of:
            DrawingWindow.HUGE, LARGE, MEDIUM, SMALL. The default is
            DrawingWindow.MEDIUM.

        >>> from cpif.graphics import DrawingWindow
        >>> dw = DrawingWindow(200, 100)
        >>> text = "What's happening?"
        >>> w, h = dw.getTextEntrySize(text)
        >>> te = dw.textEntry(0, 0, size=(w, h))
        >>> dw.setText(te, text)
        >>> dw.getText(te)
        "What's happening?"
        >>> dw.runtest()
        
        >>> dw = DrawingWindow(200, 100)
        >>> text = "ABC123"
        >>> te = dw.textEntry(0, 0, width=len(text))
        >>> dw.setText(te, text)
        >>> dw.getText(te)
        'ABC123'
        >>> dw.runtest()
        """
        if width is None and size is None:
            raise ValueError("Must specify one of width or size")
        if width is not None and size is not None:
            raise ValueError("Must specify width or size, not both")
        if keypress_callback:
            _checkParams(keypress_callback, 1)
        if self.interactive:
            self.window.update()
        sv = Tk.StringVar()
        frame = Tk.Frame(self.canvas)
        font = self._getFont(font_size)
        if prompt:
            label = Tk.Label(frame, text=prompt, font=font)
            label.pack(side=Tk.LEFT)
        if width is None:
            w = size[0]
            if prompt:
                w -= self.getTextSize(prompt, font_size=font)[0]
            w = max(w, 0)
            char_width = self.getTextSize("M", font_size=font)[0]
            char_width = max(char_width, 1)
            width = w // char_width
            if width <= 0:
                width = 1
        entry = Tk.Entry(frame, textvariable=sv, font=font, width=width)
        entry.pack(side=Tk.LEFT)
        extra_params = {}
        if size is not None:
            w, h = size
            extra_params['width'] = w
            extra_params['height'] = h
        id = self.canvas.create_window(x, y, window=frame, anchor=Tk.NW,
                                      tag=tuple(self.__tag_stack),
                                      **extra_params)
        self.__storeItemData(id, entry, self.__tag_stack)
        if set_focus:
            entry.focus_set()
        if keypress_callback:
            def _callback(event):
                if entry.focus_get() == entry:
                    if keypress_callback(KeyEvent(event)):
                        return 'break'
                    else:
                        return ''
                else:
                    return ''
            entry.bind('<KeyPress>', _callback)
        if self.interactive:
            self.window.update()
        return id


class Transform2D:
    """
    2-dimensional coordinate transformation.

    Rotate, scale, and translate coordinates to a new frame of reference.
    Algorithms from "Principles of Computer Graphics", Cottam, 1989, p. 39.
    """

    def __init__(self):
        """
        Initialize the 2-dimensional transform.
        """
        # c(mn) -> cell at row m, column n
        # column 3 is constant [0.0, 0.0, 1.0], ommitted for performance
        self.c11 = 1.0; self.c12 = 0.0
        self.c21 = 0.0; self.c22 = 1.0
        self.c31 = 0.0; self.c32 = 0.0

    def dump(self):
        print ("[[%f %f %f]\n"
               " [%f %f %f]\n"
               " [%f %f %f]]" % (self.c11, self.c12, 0.0,
                                 self.c21, self.c22, 0.0,
                                 self.c31, self.c32, 1.0))

    def mult(self, t):
        """
        t: another Transform2D

        Multiply this transform by t via matrix multiplication. 
        """
        c11 = self.c11 * t.c11 + self.c12 * t.c21 # + 0.0 * t.c31
        c12 = self.c11 * t.c12 + self.c12 * t.c22 # + 0.0 * t.c32
        c21 = self.c21 * t.c11 + self.c22 * t.c21 # + 0.0 * t.c31
        c22 = self.c21 * t.c12 + self.c22 * t.c22 # + 0.0 * t.c32
        c31 = self.c31 * t.c11 + self.c32 * t.c21 + t.c31
        c32 = self.c31 * t.c12 + self.c32 * t.c22 + t.c32
        self.c11 = c11; self.c12 = c12
        self.c21 = c21; self.c22 = c22
        self.c31 = c31; self.c32 = c32

    def rotate(self, angle):
        """
        Rotate the transformation by (angle) degrees.
        """
        # convert angle from degrees to radians
        a = angle * (math.pi / 180.0)
        cosa = math.cos(a)
        sina = math.sin(a)
        c11 = (self.c11 * cosa) - (self.c12 * sina)
        c12 = (self.c11 * sina) + (self.c12 * cosa)
        c21 = (self.c21 * cosa) - (self.c22 * sina)
        c22 = (self.c21 * sina) + (self.c22 * cosa)
        c31 = (self.c31 * cosa) - (self.c32 * sina)
        c32 = (self.c31 * sina) + (self.c32 * cosa)
        self.c11 = c11; self.c12 = c12
        self.c21 = c21; self.c22 = c22
        self.c31 = c31; self.c32 = c32

    def scale(self, xfactor, yfactor):
        """
        Scale the transformation by xfactor, yfactor.
        """
        self.c11 = self.c11 * xfactor
        self.c12 = self.c12 * yfactor
        self.c21 = self.c21 * xfactor
        self.c22 = self.c22 * yfactor
        self.c31 = self.c31 * xfactor
        self.c32 = self.c32 * yfactor

    def set(self, angle, xscale, yscale, xoffset, yoffset):
        """
        Set the 2-D transformation by rotating, scaling, and translating,
        in that order.

        angle: Rotation angle in degrees
        xscale, yscale: Amount to scale coordinates
        xoffset, yoffset: Origin point for translating coordinates
        """
        # convert angle from degrees to radians
        a = angle * (math.pi / 180.0)
        # generate matrix coefficents
        self.c11 = math.cos(a) * xscale
        self.c12 = math.sin(a) * yscale
        self.c21 = -math.sin(a) * xscale
        self.c22 = math.cos(a) * yscale
        self.c31 = xoffset
        self.c32 = yoffset

    def transform(self, points):
        """
        points: [(x, y), ...]
        Return a list of transformed points.
        """
        c11 = self.c11; c12 = self.c12
        c21 = self.c21; c22 = self.c22
        c31 = self.c31; c32 = self.c32
        result = []
        for x, y in points:
            x1 = x * c11 + y * c21 + c31
            y1 = x * c12 + y * c22 + c32
            result.append((x1, y1))
        return result

    def translate(self, xoffset, yoffset):
        """
        Translate the 2-D transformation by (xoffset, yoffset).
        """
        self.c31 = self.c31 + xoffset
        self.c32 = self.c32 + yoffset


################################################################
# internal functions, classes and data


_COLOR_USAGE_MSG = (
    "Color parameter must be either a string containing a color name\n"
    "or a tuple of (red, green, blue) values. The values must be\n"
    "between 0 and 255, inclusive.")

_VALID_COLOR_RANGE = xrange(256)

def _convertColor(colorparam):
    """
    Convert a color from the (r, g, b) tuple representation to the
    '#rrggbb' style string that Tk uses. If the color parameter is
    already a string, return it unchanged.

    The r, g, and b numbers are expected to be integers between 0
    and 255, inclusive.

    This method attempts to be robust in the face of user mistakes.
    It raises a ValueError with a usage message for all invalid
    non-string parameters.
    """
    if isinstance(colorparam, basestring):
        return colorparam
    try:
        red, green, blue = map(int, colorparam) # [int(p) for p in colorparam]
        validrange = _VALID_COLOR_RANGE
        if red not in validrange or green not in validrange or \
           blue not in validrange:
            raise ValueError()
        return '#%2.2x%2.2x%2.2x' % (red, green, blue)
    except (TypeError, ValueError):
        raise ValueError(_COLOR_USAGE_MSG)


_CHECK_PARAMS_MSG = (
    "callback must be a function or method that takes %d parameters")

_CHECK_KWPARAMS_MSG = (
    "callback doesn't support keyword parameter: '%s'")

_CHECK_KWPARAMMULT_MSG = (
    "keyword parameter '%s' gets multiple values")

def _checkParams(callback, nparams, kwparams=()):
    """
    Raise a ValueError if the callback is not a callable object or it is
    known that it cannot take the required parameters.

    Built-in functions can't be checked for the number of parameters, but
    nearly every other callable object can be.
    """
    if not callable(callback):
        raise ValueError(_CHECK_PARAMS_MSG % nparams)
    if inspect.isbuiltin(callback):
        # no further checking possible
        return
    if inspect.isfunction(callback):
        c = callback
        bound = False
    elif inspect.ismethod(callback):
        c = callback
        bound = callback.im_self is not None
    elif inspect.isclass(callback):
        if not hasattr(callback, '__init__'):
            if nparams or kwparams:
                raise ValueError(_CHECK_PARAMS_MSG % nparams)
            else:
                return
        c = callback.__init__
        bound = True # __init__ will be bound at the time it is called
    else:
        # Some other callable object that I don't know how to check.
        return
    args, varargs, varkw, defaults = inspect.getargspec(c)
    required_params = len(args)
    if bound:
        required_params -= 1
    allowed_params = required_params
    if defaults:
        required_params -= len(defaults)
        defined_kwparams = args[-len(defaults):]
    else:
        defined_kwparams = ()
    if varargs:
        # No upper bound on the number of parameters
        allowed_params = sys.maxint
    #print "required_params",required_params,"allowed_params",allowed_params
    if nparams < required_params or nparams > allowed_params:
        raise ValueError(_CHECK_PARAMS_MSG % nparams)
    if not kwparams:
        # No keyword parameters to check, our job is done
        return
    # Check for multiple values for defined kw params
    num_kwparams_used = nparams - required_params
    if num_kwparams_used > 0:
        for used_kwparam in defined_kwparams[:num_kwparams_used]:
            if used_kwparam in kwparams:
                raise ValueError(_CHECK_KWPARAMMULT_MSG % used_kwparam)
    if varkw:
        # Any kw params allowed, our job is done
        return
    # Check for use of undefined kw params
    for kwparam in kwparams:
        if kwparam not in defined_kwparams:
            raise ValueError(_CHECK_KWPARAMS_MSG % kwparam)


class Event(object):
    """
    An Event base class, so we can easily test if an object is a 
    KeyEvent or MouseEvent with the following code:
    
    isinstance(obj, Event)
    """

    def __init__(self, event):
        """
        Initialize the base class with a Tkinter event object.
        """
        # Save the original Tkinter event for advanced users.
        # This is an undocumented feature, it may go away in the next
        # version. You have been warned!
        self._event = event

    def __str__(self):
        names = [name for name in dir(self) if name != 'event' and 
                    not name.startswith('_')]
        items = zip(names, [getattr(self, name) for name in names])
        return ' '.join(
                [self.__class__.__name__] + 
                ["%s=%s" % (name, repr(value)) for name, value in items])


class KeyEvent(Event):
    """
    An instance of this class is the key_event parameter passed to the
    callback set by onKeyDownCall() and onKeyUpCall().

    The attributes are:
        char: The keyboard character as a string.
        keysym: The keyboard symbol of the character.
    """

    def __init__(self, event):
        """
        Initialize a KeyEvent from a Tkinter event.
        """
        super(KeyEvent, self).__init__(event)
        self.char = event.char
        self.keysym = event.keysym


class MouseEvent(Event):
    """
    An instance of this class is the mouse_event parameter passed to the
    callback set by onMouseDownCall(), onMouseUpCall(), and
    onMouseMoveCall().

    The attributes are:
        button_num:
            The number of the mouse button pressed, or 0 for no button
            (i.e. for an event sent when the mouse is moved)
            1 = left button, 2 = center button, 3 = right button
        x: the X coordinate, in window pixels, of the mouse cursor
        y: the Y coordinate, in window pixels, of the mouse cursor
    """

    def __init__(self, event):
        """
        Inialize a MouseEvent from a Tkinter event.
        """
        super(MouseEvent, self).__init__(event)
        self.button_num = event.num
        self.x = event.x
        self.y = event.y


class Timer:
    """
    A timer object that calls a function at regular intervals.

    A Timer object is created by calling DrawingWindow.onTimerCall().
    """

    def __init__(self, dw, interval_sec, callback, one_shot=False):
        """
        dw: a DrawingWindow object
        interval_sec: The time interval of the timer, in seconds.
        callback: The function that the timer will call.
        one_shot: If true, only execute the callback function once
                  (unless start() is called again.) Default is False.
        """
        self.dw = dw
        self.interval_sec = interval_sec
        self.callback = callback
        self.one_shot = one_shot
        self.__lock = threading.RLock()
        self.__timer_id = None

    def start(self):
        self.__lock.acquire()
        try:
            if not self.__timer_id:
                delay_ms = int(self.interval_sec * 1000.)
                self.__timer_id = self.dw.window.after(delay_ms, self._callback)
        finally:
            self.__lock.release()

    def stop(self):
        self.__lock.acquire()
        try:
            timer_id = self.__timer_id
            self.__timer_id = None
        finally:
            self.__lock.release()
        if timer_id:
            self.dw.window.after_cancel(timer_id)

    # Return the time with the most precision available, depending on
    # the platform.
    if sys.platform.startswith('win'):
        _time = time.clock
    else:
        _time = time.time

    def _callback(self):
        # Don't execute the callback if the timer has already been stopped.
        # The lock guarentees that self.__timer_id will be set properly before
        # we check it.
        self.__lock.acquire()
        try:
            if not self.__timer_id:
                return
        finally:
            self.__lock.release()
        start_time = self._time()
        self.callback()
        self.__lock.acquire()
        try:
            if self.one_shot:
                self.__timer_id = None
            elif self.__timer_id:
                delay = max(0, self.interval_sec - (self._time() - start_time))
                self.__timer_id = self.dw.window.after(int(delay * 1000.), 
                                                       self._callback)
        finally:
            self.__lock.release()


class GraphFunc:
    """
    Graph a function over a range of values.
    """

    def __init__(self, f, xmin, xmax, xdelta, ymin, ymax, width, height):
        self.f = f
        self.xmin = xmin
        self.xmax = xmax
        self.xdelta = xdelta
        self.ymin = ymin
        self.ymax = ymax
        self.width = width
        self.height = height
        self.dw = DrawingWindow(width, height)
        self._drawGraph()

    def run(self):
        return self.dw.run()

    def _drawGraph(self):
        f = self.f
        xmin = self.xmin
        xmax = self.xmax
        xdelta = self.xdelta
        ymin = self.ymin
        ymax = self.ymax
        width = self.width
        height = self.height
        dw = self.dw
        x = xmin
        while x <= xmax:
            y = f(x)
            px = (float(x - xmin) * width) / (xmax - xmin)
            py = (float(ymax - y) * height) / (ymax - ymin)
            dw.box(px, py, px + 1, py + 1, 'black')
            x += xdelta


__test__ = {
    '_checkParams':
    r"""
    >>> def f0():
    ...     pass
    >>> _checkParams(f0, 0)
    >>> _checkParams(f0, 0, {})
    >>> _checkParams(f0, 1)
    Traceback (most recent call last):
      ...
    ValueError: callback must be a function or method that takes 1 parameters
    >>> _checkParams(f0, 0, {'kw1': 1})
    Traceback (most recent call last):
      ...
    ValueError: callback doesn't support keyword parameter: 'kw1'

    >>> def f1(a):
    ...     pass
    >>> _checkParams(f1, 1)
    >>> _checkParams(f1, 1, {})
    >>> _checkParams(f1, 0)
    Traceback (most recent call last):
      ...
    ValueError: callback must be a function or method that takes 0 parameters
    >>> _checkParams(f1, 2)
    Traceback (most recent call last):
      ...
    ValueError: callback must be a function or method that takes 2 parameters
    >>> _checkParams(f1, 1, {'kw1': 1})
    Traceback (most recent call last):
      ...
    ValueError: callback doesn't support keyword parameter: 'kw1'

    >>> def f1_1(a, kw1=None):
    ...     pass
    >>> _checkParams(f1_1, 1, {'kw1':1})
    >>> _checkParams(f1_1, 2)
    >>> _checkParams(f1_1, 0, {'kw1':1})
    Traceback (most recent call last):
      ...
    ValueError: callback must be a function or method that takes 0 parameters
    >>> _checkParams(f1_1, 1, {'xyz':1})
    Traceback (most recent call last):
      ...
    ValueError: callback doesn't support keyword parameter: 'xyz'
    >>> _checkParams(f1_1, 1, {'kw1':1, 'xyz':1})
    Traceback (most recent call last):
      ...
    ValueError: callback doesn't support keyword parameter: 'xyz'

    >>> def f1_1n(a, kw1=None, **kwparams):
    ...     pass
    >>> _checkParams(f1_1n, 1, {'xyz':'abc'})
    >>> _checkParams(f1_1n, 0)
    Traceback (most recent call last):
      ...
    ValueError: callback must be a function or method that takes 0 parameters
    >>> _checkParams(f1_1n, 2, {'kw1':'abc'})
    Traceback (most recent call last):
      ...
    ValueError: keyword parameter 'kw1' gets multiple values

    >>> class C:
    ...     def m(self, dw):
    ...         pass
    >>> c = C()
    >>> _checkParams(c.m, 1)
    >>> _checkParams(c.m, 1, {})
    >>> _checkParams(c.m, 0)
    Traceback (most recent call last):
      ...
    ValueError: callback must be a function or method that takes 0 parameters
    >>> _checkParams(c.m, 1, {'kw1': 1})
    Traceback (most recent call last):
      ...
    ValueError: callback doesn't support keyword parameter: 'kw1'
    """,

    '_getFont_standard_font_sizes':
    """
    >>> dw = DrawingWindow(400, 400)
    >>> s = 'ABCabc'
    >>> t1 = dw.drawText(0, 0, s, font_size=DrawingWindow.HUGE)
    >>> t2 = dw.drawText(0, 100, s, font_size=DrawingWindow.LARGE)
    >>> t3 = dw.drawText(0, 200, s, font_size=DrawingWindow.MEDIUM)
    >>> t4 = dw.drawText(0, 300, s, font_size=DrawingWindow.SMALL)
    >>> w1, h1 = dw.getTextSize(s, font_size=DrawingWindow.HUGE)
    >>> w2, h2 = dw.getTextSize(s, font_size=DrawingWindow.LARGE)
    >>> w3, h3 = dw.getTextSize(s, font_size=DrawingWindow.MEDIUM)
    >>> w4, h4 = dw.getTextSize(s, font_size=DrawingWindow.SMALL)
    >>> w, h = dw.getTextSize(s)
    >>> assert w == w3
    >>> assert h == h3
    >>> assert w1 > w2
    >>> assert w2 > w3
    >>> assert w3 > w4
    >>> assert w4 > 0
    >>> dw.runtest()
    >>> dw.close()
    """,

    '_getFont_negative_font_size':
    """
    >>> dw = DrawingWindow(200, 100)
    >>> M = 'M'
    >>> # Negative font size == absolute pixels
    >>> sizes = (-20, -40, -60, -80)
    >>> tx, ty = 0, 0
    >>> #import sys
    >>> for size in sizes:
    ...     w, h = dw.getTextSize(M, font_size=size)
    ...     t = dw.drawText(tx, ty, M, font_size=size)
    ...     #print >> sys.stderr, "M size (%d):" % size, w, h
    ...     tx = tx + w
    ... 
    >>> dw.runtest()
    >>> dw.close()
    """,

    'startup_callback':
    """
    >>> message = []
    >>> def onStartup():
    ...     message.append("Hello from onStartup")
    ...
    >>> dw = DrawingWindow(100, 100, 'red', startup_callback=onStartup)
    >>> dw.runtest()
    >>> print message[0]
    Hello from onStartup
    >>> dw.close()
    """,
    }


if __name__ == '__main__':
    import doctest
    doctest.testmod()

# end-of-file
