# fullscreen.py
# David Handy  23 Feb 2006
"""
Create Tkinter main window capable of going "fullscreen"
"""
# Status:
# 
# Startup up in fullscreen mode and receiving keyboard events works, after
# applying workarounds for the X window system.
# 
# Notes:
# 
# On MS Windows everything works perfectly. I don't have to do a
# grab_set_global(), and I can switch back and forth between fullscreen and
# normal mode at will, as many times as I want.
# 
# On the X window system on Linux, I had to put in lots of workarounds.  I
# wasn't getting any keyboard events in the fullscreen window unless I do a
# grab_set_global(). I tried everything else I could think of, including
# regular grab_set(), and nothing worked. Even though there is an Entry
# widget and it visibly has focus and a cursor, when you type nothing
# happens, unless you do a grab_set_global().  If I do a grab_release(),
# immediately I stop getting keyboard events. (This, and other workarounds,
# have been incorporated into this code.)
# 
# Also on the X window system I had to put in code to detect that the
# switch to fullscreen mode really suceeded, and if not, retry it.
# Otherwise I would end up with a missing window and a hung application.
# 
# TODO:
# 
# Research the bad effects of grab_set_global() on the X window system,
# particularly when a screensaver tries to activate.
# 
# Sources:
# http://effbot.org/zone/tkinter-toplevel-fullscreen.htm
# http://mail.python.org/pipermail/python-list/1999-November/015868.html
# http://wiki.tcl.tk/12506
# http://www.binarism.com/tk/window/or/

import logging
import re
import sys
import Tkinter as Tk
import threading

log = logging.getLogger('cpif.fullscreen')


class FullScreenTk(Tk.Tk):

    """
    Toplevel Tk widget which represents the main window of an application
    that is capable of switching between fullscreen and normal modes.
    """

    def __init__(self, default_width=640, default_height=480, 
                       default_x=150, default_y=100, **kwparams):
        """
        default_width, default_height, default_x, default_y -- 
            If fullscreen() is called before mainloop() is called, then the
            application starts up in fullscreen mode. In that case, the
            default width, height, x, and y parameters are used to set the
            window size and position later when normal() is called for the
            first time."""
        Tk.Tk.__init__(self, **kwparams)
        self.__last_width, self.__last_height = (default_width, default_height)
        self.__last_x, self.__last_y = (default_x, default_y)
        self.__need_global_grab = False
        self.__valid_size_and_pos = False
        self.__max_width, self.__max_height = self.wm_maxsize()
        self.__fullscreen_followup_timer = None
        self.__watchdog_timer = None
        # Synchronized attributes:
        self.__mode_change_lock = threading.RLock()
        self.__mode_current = self._MODE_NORMAL
        self.__mode_next = None
        self.__mode_change = False
        # Trap first expose event
        self.bind('<Expose>', self._on_expose)
        # Avoid the annoying "No handlers could be found for logger" message
        logging.basicConfig()

    def fullscreen(self):
        log.info("fullscreen")
        self._modeChangeRequest(self._MODE_FULLSCREEN)

    def isfullscreen(self):
        return self.__mode_current == self._MODE_FULLSCREEN

    def normal(self):
        log.info("normal")
        self._modeChangeRequest(self._MODE_NORMAL)

    def switchmode(self):
        log.info("switchmode")
        self._modeChangeRequest()

    _WATCHDOG_TIMEOUT_MS = 2000
    _FULLSCREEN_FOLLOWUP_TIMEOUT_MS = 1
    _RETRY_MS = 200

    def _modeChangeRequest(self, next_mode=None):
        log.debug("_modeChangeRequest(%s)", next_mode)
        next_action = None # default: no mode change
        mode_current = self.__mode_current
        self.__mode_change_lock.acquire()
        try:
            if next_mode is None:
                # Switch modes
                # Recursive call works because we're using RLock instead of
                # Lock
                if mode_current == self._MODE_NORMAL:
                    return self._modeChangeRequest(self._MODE_FULLSCREEN)
                else:
                    return self._modeChangeRequest(self._MODE_NORMAL)
            if self.__mode_change:
                # Mode was already changing
                if self.__mode_next:
                    # There is a already another mode change waiting
                    if self.__mode_next != next_mode:
                        # Countermand next mode change
                        self.__mode_next = None
                else:
                    # Put this mode change on the waiting list
                    self.__mode_next = next_mode
            else:
                if self.__mode_current != next_mode:
                    self.__mode_current = next_mode
                    self.__mode_change = True
                    next_action = self._MODE_ACTIONS[next_mode]
        finally:
            self.__mode_change_lock.release()
        self._modeDoAction(next_action)

    def _modeDoAction(self, action, *params):
        if action:
            action(self, *params)
            self.bind('<Expose>', self._on_expose)
            self.__watchdog_timer = self.after(self._WATCHDOG_TIMEOUT_MS, 
                                               self._on_watchdog_timer)

    def _modeCancel(self):
        # This is only called in case of an unrecoverable error, to avoid an
        # infinite loop of events begetting other events. The window is not
        # guaranteed to be in any particular state when this is done.
        self.__mode_change = False
        self.unbind('<Expose>')
        if self.__watchdog_timer:
            self.after_cancel(self.__watchdog_timer)
            self.__watchdog_timer = None

    def _modeRetry(self, *params):
        next_action = self._MODE_ACTIONS[self.__mode_current]
        self._modeDoAction(next_action, *params)

    def _modeChangeComplete(self):
        log.debug("_modeChangeComplete")
        next_action = None
        self.__mode_change_lock.acquire()
        try:
            if self.__mode_next:
                self.__mode_current = self.__mode_next
                self.__mode_next = None
                next_action = self._MODE_ACTIONS[self.__mode_current]
            else:
                self.__mode_change = False
        finally:
            self.__mode_change_lock.release()
        self._modeDoAction(next_action)

    def _fullscreen(self, save_size_and_pos=True):
        log.debug("start switch to fullscreen")
        if save_size_and_pos and self.__valid_size_and_pos:
            # Save window size and position, if they are valid
            # (the window might never have been visible).
            m = re.match(r'(\d+)x(\d+)\+(\d+)\+(\d+)', self.wm_geometry())
            if m:
                self.__last_width = int(m.group(1))
                self.__last_height = int(m.group(2))
                self.__last_x = int(m.group(3))
                self.__last_y = int(m.group(4))
        width, height = self.winfo_screenwidth(), self.winfo_screenheight()
        self.wm_maxsize(width=width, height=height)
        self.withdraw()
        self.overrideredirect(1)
        self.deiconify()
        self.lift()
        self.wm_geometry("%dx%d+0+0" % (width, height))
        self.update_idletasks()
        wm_attrs = self.wm_attributes()
        log.debug("wm_attrs: %s", repr(wm_attrs))
        try:
            self.wm_attributes('-topmost', 'yes')
            self.__need_global_grab = False
        except Tk.TclError, e:
            log.debug('"topmost" wm attribute not supported, need global grab')
            self.__need_global_grab = True
        log.debug("end switch to fullscreen")

    def _normal(self):
        log.debug("start switch to normal mode")
        width, height = self.__last_width, self.__last_height
        x, y = self.__last_x, self.__last_y
        self.wm_maxsize(width=self.__max_width, height=self.__max_height)
        self.withdraw()
        self.overrideredirect(0)
        self.wm_geometry("%dx%d+%d+%d" % (width, height, x, y))
        self.grab_release()
        self.deiconify()
        self.wait_visibility()
        try:
            self.wm_attributes('-topmost', 'no')
        except Tk.TclError, e:
            pass
        self.lift()
        self.focus_set()
        log.debug("end switch to normal mode")

    def _on_expose(self, event):
        if event and self != event.widget:
            return ''
        log.debug("_on_expose")
        self.unbind('<Expose>')
        if self.__watchdog_timer:
            self.after_cancel(self.__watchdog_timer)
            self.__watchdog_timer = None
        self._MODE_EXPOSE_ACTIONS[self.__mode_current](self, event)

    def _on_fullscreen_expose(self, event):
        log.debug("_on_fullscreen_expose begin")
        if self.wm_state() == 'iconic':
            log.warn('_on_fullscreen_expose: still iconic, trying again')
            self.after(self._RETRY_MS, self._modeRetry, False)
            return ''
        if self.winfo_geometry() != self.wm_geometry():
            log.warn('_on_fullscreen_expose: '
                     'window not in place, trying again')
            self.after(self._RETRY_MS, self._modeRetry, False)
            return ''
        #self.wm_overrideredirect(1)
        if self.__need_global_grab:
            retry_grab = False
            try:
                self.grab_set_global()
            except Tk.TclError:
                t, e, tb = sys.exc_info()
                log.warn("_on_fullscreen_expose: %s", e)
                if 'another application has grab' in str(e):
                    # TclError: grab failed: another application has grab
                    # Can't recover
                    self._modeCancel()
                    raise t, e, tb
                del tb
                # TclError: grab failed: window not viewable
                retry_grab = True
            else:
                retry_grab = (self != self.grab_current())
            if retry_grab:
                log.warn('_on_fullscreen_expose: grab failed, trying again')
                self.grab_release() # release helps trying again?
                self.after(self._RETRY_MS, self._modeRetry, False)
                return ''
            else:
                log.debug("global event grab succeeded")
                log.debug("grab_current(): %s", self.grab_current())
                log.debug("grab_status(): %s", self.grab_status())
        self.focus_force()
        log.debug("focus_get(): %s", self.focus_get())
        if self != self.focus_get():
            log.warn('_on_fullscreen_expose: focus failed, trying again')
            self.after(self._RETRY_MS, self._modeRetry, False)
            return ''
        self.__fullscreen_followup_timer = self.after(
                self._FULLSCREEN_FOLLOWUP_TIMEOUT_MS,
                self._on_fullscreen_followup_timer)
        log.debug("_on_fullscreen_expose end")

    def _on_fullscreen_followup_timer(self):
        log.debug("_on_fullscreen_followup_timer")
        log.debug("focus_get(): %s", self.focus_get())
        self.__fullscreen_followup_timer = None
        if self != self.focus_get():
            log.warn('_on_fullscreen_followup_timer: '
                     'focus failed, trying again')
            self.focus_force()
            self.__fullscreen_followup_timer = self.after(
                    self._FULLSCREEN_FOLLOWUP_TIMEOUT_MS,
                    self._on_fullscreen_followup_timer)
            return
        self._modeChangeComplete()

    def _on_normal_expose(self, event):
        log.debug("_on_normal_expose begin")
        self.__valid_size_and_pos = True
        self._modeChangeComplete()
        log.debug("_on_normal_expose end")

    def _on_watchdog_timer(self):
        log.debug("***** _on_watchdog_timer *****")
        log.warn("No expose event after %d milliseconds", 
                self._WATCHDOG_TIMEOUT_MS)
        self._trace_info()
        self.__watchdog_timer = None
        log.debug('_on_watchdog_timer: retrying %s', self.__mode_current)
        self.after(0, self._on_expose, None)

    def _trace_info(self, info=log.info):
        info("---------- trace window info ----------")
        info("  focus_get(): %s", self.focus_get())
        info("  grab_current(): %s", self.grab_current())
        info("  grab_status(): %s", self.grab_status())
        info("  winfo_viewable(): %s", self.winfo_viewable())
        info("  winfo_width(), winfo_height(): %sx%s",
                self.winfo_width(), self.winfo_height())
        info("  winfo_x(), winfo_y(): +%s+%s", self.winfo_x(),
                self.winfo_y())
        info("  winfo_geometry(): %s", self.winfo_geometry())
        info("  wm_state(): %s", self.wm_state())
        info("  wm_geometry(): %s", self.wm_geometry())


    _MODE_NORMAL = 'normal'
    _MODE_FULLSCREEN = 'fullscreen'
    _MODE_ACTIONS = {_MODE_FULLSCREEN: _fullscreen,
                     _MODE_NORMAL: _normal}
    _MODE_EXPOSE_ACTIONS = {_MODE_FULLSCREEN: _on_fullscreen_expose,
                            _MODE_NORMAL: _on_normal_expose}


def main():
    def keypress(event):
        print "KeyPress char:", repr(event.char), "keysym:", repr(event.keysym)
        sys.stdout.flush()
        if event.keysym == 'Escape':
            root.quit()
            return 'break'
        if event.keysym == 'F10':
            root.switchmode()
            return 'break'
        return ''
    logging.basicConfig(format="%(message)s")
    log.setLevel(logging.DEBUG)
    root = FullScreenTk()
    root.configure(borderwidth=5, bg='blue')
    label = Tk.Label(root, 
        text='Esc to quit, F10 to switch between normal window and fullscreen',
        bg='blue', fg='white')
    label.pack(padx=10, pady=10)
    entry = Tk.Entry(root)
    entry.pack(padx=10, pady=10)
    button = Tk.Button(root, text='Switch modes', command=root.switchmode)
    button.pack(side=Tk.TOP)
    def dump_info():
        def writeline(format, *params):
            sys.stdout.write(format % params)
            sys.stdout.write('\n')
            sys.stdout.flush()
        root._trace_info(info=writeline)
    button = Tk.Button(root, text='Log info', command=dump_info)
    button.pack(side=Tk.TOP)
    button = Tk.Button(root, text='Quit', command=root.quit)
    button.pack(side=Tk.TOP)
    if '--fullscreen' in sys.argv[1:]:
        root.fullscreen()
    if '--grab' in sys.argv[1:]:
        root.grab_set_global()
    root.bind('<KeyPress>', keypress)
    root.mainloop()

if __name__ == '__main__':
    main()

# end-of-file
