# lander6.py
# David Handy  June 2004
"""
Classic Lunar-lander style game.

Use the arrow keys to fire the rockets, Esc to quite the game.
"""

from cpif import sound, music
from cpif.graphics import DrawingWindow


REFRESH_INTERVAL = 0.05 # seconds between updates


class SpaceShip:
    
    def __init__(self, startx, starty):
        self.startx = startx
        self.starty = starty
        self.cur_sound = None
        self.x = self.startx
        self.y = self.starty
        self.size = 20
        self.id = self.drawShip(self.x, self.y)
        self.init()

    def init(self):
        self.vx = 0.0
        self.vy = 0.0
        self.right_flame = None
        self.left_flame = None
        self.down_flame = None
        dw.moveBy(self.id, self.startx - self.x, self.starty - self.y)
        self.x = self.startx
        self.y = self.starty
        self.intro_text = self.drawTextAboveShip(
                              "Press the arrow keys to fire the rockets")
        self.landed = None
        self.crashed = None
        start_sound = music.soundOfMusic([('E4', 0.125), ('C4', 0.125)])
        start_sound.play()

    def keyUpCallback(self, event):
        if event.keysym == 'Down':
            if self.down_flame is not None:
                dw.delete(self.down_flame)
                self.down_flame = None
        elif event.keysym == 'Right':
            if self.right_flame is not None:
                dw.delete(self.right_flame)
                self.right_flame = None
        elif event.keysym == 'Left':
            if self.left_flame is not None:
                dw.delete(self.left_flame)
                self.left_flame = None

    def keyDownCallback(self, event):
        if self.intro_text:
            dw.delete(self.intro_text)
            self.intro_text = None
        if event.keysym == 'Escape':
            dw.close()
        elif event.keysym == 'Down':
            if not self.down_flame:
                self.drawDownFlame()
        elif event.keysym == 'Right':
            if not self.right_flame:
                self.drawSideFlame(1)
        elif event.keysym == 'Left':
            if not self.left_flame:
                self.drawSideFlame(-1)
        elif event.char.upper() == 'R':
            # restart
            for id in (self.down_flame, self.right_flame, self.left_flame,
                       self.intro_text, self.landed, self.crashed):
                if id:
                    dw.delete(id)
            self.init()

    def drawShip(self, x, y):
        size = self.size
        p1 = (x,          y - size)
        p2 = (x + size/2, y + size)
        p3 = (x,          y + size/2)
        p4 = (x - size/2, y + size)
        id = dw.polygon([p1, p2, p3, p4], 'black', fill='blue')
        return id

    def drawSideFlame(self, direction):
        size = self.size
        x, y = self.x, self.y
        p1 = (x + direction*size*.25, y)
        p2 = (x + direction*size*.50, y - size*.25)
        p3 = (x + direction*size*.75, y)
        p4 = (x + direction*size*.50, y + size*.25)
        id = dw.polygon([p1, p2, p3, p4], 'red', fill='orange')
        if direction > 0:
            self.right_flame = id
        else:
            self.left_flame = id

    def drawDownFlame(self):
        size = self.size
        x, y = self.x, self.y
        p1 = (x,            y + size    )
        p2 = (x + size*.25, y + size*1.5)
        p3 = (x,            y + size*2.0)
        p4 = (x - size*.25, y + size*1.5)
        id = dw.polygon([p1, p2, p3, p4], 'red', fill='orange')
        self.down_flame = id

    def update(self):
        if self.intro_text or self.landed or self.crashed:
            return
        # acceleration
        gravity = 0.07
        side_rocket_accel = 0.03
        down_rocket_accel = 0.14
        if self.right_flame:
            self.vx = self.vx - side_rocket_accel
        if self.left_flame:
            self.vx = self.vx + side_rocket_accel
        if self.down_flame:
            self.vy = self.vy - down_rocket_accel
        self.vy = self.vy + gravity
        # update position
        self.x = self.x + self.vx
        self.y = self.y + self.vy
        for id in (self.id, self.right_flame, self.left_flame,
                   self.down_flame):
            if id:
                dw.moveBy(id, self.vx, self.vy)
        # detect crash or landing
        crash_speed = 0.50
        bottom_of_ship = self.y + self.size
        window_width, window_height = dw.getSize()
        if bottom_of_ship > window_height:
            if self.vy > crash_speed:
                message = "You crashed! Press R to restart, Esc to quit"
                self.crashed = self.drawTextAboveShip(message)
                self.playSound('boom.wav')
            else:
                message = "Safe landing! Press R to restart, Esc to quit"
                self.landed = self.drawTextAboveShip(message)
                land_music = [('C4', 0.125), ('D4', 0.125), ('E4', 0.125), 
                              ('E5', 0.25), ('C5', 0.125)]
                land_sound = music.soundOfMusic(land_music)
                land_sound.play()
            return
        # sound effects
        busy = self.cur_sound is not None and self.cur_sound.isPlaying()
        if self.crashed:
            self.playSound('boom.wav')
        elif self.down_flame:
            if not busy:
                self.playSound('thrust2.wav')
        elif self.left_flame or self.right_flame:
            if not busy:
                self.playSound('thrust1.wav')

    def drawTextAboveShip(self, message):
        """
        Draw text centered above the top of the space ship.
        Return the id of the newly-created text.
        """
        text_width, text_height = dw.getTextSize(message)
        text_x = self.x - (text_width / 2)
        text_y = self.y - self.size - text_height
        id = dw.drawText(text_x, text_y, message)
        return id

    def playSound(self, soundfile):
        """
        Set the current sound and start it playing
        """
        self.cur_sound = sound.Sound(soundfile)
        self.cur_sound.play()


width, height = 640, 480
dw = DrawingWindow(width, height)
ship = SpaceShip(width/2, height/2)
dw.onKeyDownCall(ship.keyDownCallback)
dw.onKeyUpCall(ship.keyUpCallback)
dw.onTimerCall(REFRESH_INTERVAL, ship.update)
dw.run()

# end-of-file
