# lander4.py
# David Handy  June 2004
"""
Classic Lunar-lander style game.

Use the arrow keys to fire the rockets, Esc to quite the game.
"""

from cpif.graphics import DrawingWindow


REFRESH_INTERVAL = 0.05 # seconds between updates


class SpaceShip:
    
    def __init__(self, startx, starty):
        self.x = startx
        self.y = starty
        self.vx = 0.0
        self.vy = 0.0
        self.size = 20
        self.id = self.drawShip(self.x, self.y)
        self.right_flame = None
        self.left_flame = None
        self.down_flame = None

    def keyUpCallback(self, event):
        if event.keysym == 'Escape':
            dw.close()
        elif event.keysym == 'Down':
            if self.down_flame is not None:
                dw.delete(self.down_flame)
                self.down_flame = None
        elif event.keysym == 'Right':
            if self.right_flame is not None:
                dw.delete(self.right_flame)
                self.right_flame = None
        elif event.keysym == 'Left':
            if self.left_flame is not None:
                dw.delete(self.left_flame)
                self.left_flame = None

    def keyDownCallback(self, event):
        if event.keysym == 'Escape':
            dw.close()
        elif event.keysym == 'Down':
            self.drawDownFlame()
        elif event.keysym == 'Right':
            self.drawSideFlame(1)
        elif event.keysym == 'Left':
            self.drawSideFlame(-1)

    def drawShip(self, x, y):
        size = self.size
        p1 = (x,          y - size)
        p2 = (x + size/2, y + size)
        p3 = (x,          y + size/2)
        p4 = (x - size/2, y + size)
        id = dw.polygon([p1, p2, p3, p4], 'black', fill='blue')
        return id

    def drawSideFlame(self, direction):
        size = self.size
        x, y = self.x, self.y
        p1 = (x + direction*size*.25, y)
        p2 = (x + direction*size*.50, y - size*.25)
        p3 = (x + direction*size*.75, y)
        p4 = (x + direction*size*.50, y + size*.25)
        id = dw.polygon([p1, p2, p3, p4], 'red', fill='orange')
        if direction > 0:
            self.right_flame = id
        else:
            self.left_flame = id

    def drawDownFlame(self):
        size = self.size
        x, y = self.x, self.y
        p1 = (x,            y + size    )
        p2 = (x + size*.25, y + size*1.5)
        p3 = (x,            y + size*2.0)
        p4 = (x - size*.25, y + size*1.5)
        id = dw.polygon([p1, p2, p3, p4], 'red', fill='orange')
        self.down_flame = id

    def update(self):
        self.x = self.x + self.vx
        self.y = self.y + self.vy
        for id in (self.id, self.right_flame, self.left_flame,
                   self.down_flame):
            if id:
                dw.moveBy(id, self.vx, self.vy)
        # acceleration
        gravity = 0.07
        side_rocket_accel = 0.03
        down_rocket_accel = 0.14
        if self.right_flame:
            self.vx = self.vx - side_rocket_accel
        if self.left_flame:
            self.vx = self.vx + side_rocket_accel
        if self.down_flame:
            self.vy = self.vy - down_rocket_accel
        self.vy = self.vy + gravity


width, height = 640, 480
dw = DrawingWindow(width, height)
ship = SpaceShip(width/2, height/2)
dw.onKeyDownCall(ship.keyDownCallback)
dw.onKeyUpCall(ship.keyUpCallback)
dw.onTimerCall(REFRESH_INTERVAL, ship.update)
dw.run()

# end-of-file
